// ColorChooser.java - a Java colors chooser 
//
// Copyright (C) 1996 by Vladimir Bulatov <V.Bulatov@ic.ac.uk>.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.


import java.awt.*;
import java.awt.image.MemoryImageSource;

class ColorModel extends Object {
  public Color color = new Color(128,128,128);
  static final int R = 0, G = 1, B = 2, H = 3, S = 4, L = 5;
  public int model = R;
  public int grid = 3;
  float[] hsl = new float[3];

  ColorModel(Color color){
    this.color = color;
    makeHSL();
  }

  ColorModel(Color color,int model, int grid){
    this.color = color;
    this.model = model;
    this.grid = grid;
    makeHSL();
  }  

  void makeHSL(){
    Color.RGBtoHSB(color.getRed(),color.getGreen(),color.getBlue(),hsl);
  }

  void setColor(Color color){
    this.color = color;
    makeHSL();    
  }
}

class ColorPanel extends Canvas {

  ColorModel colorModel = null;

  public ColorPanel(ColorModel colorModel) {
    this.colorModel = colorModel;
  }

  public Dimension preferredSize(){
    return new Dimension(80,60);
  }

  public Dimension minimumSize(){
    return preferredSize();
  }

  public void update(Graphics g){
    paint(g);
  }

  public void paint(Graphics g) {
    int bytes[] = {colorModel.color.getRGB()};
    g.drawImage(this.createImage(new MemoryImageSource(1,1,bytes,0,1)),
		0,0,size().width,size().height,null);
  }  
  
}

class ColorArea extends Canvas {

  ColorModel colorModel = null;
  ColorChooser parent;
  public ColorArea(ColorModel colorModel,ColorChooser parent) {
    this.colorModel = colorModel;
    this.parent = parent;
    makeImage();    
  }

  public Dimension preferredSize(){
    return new Dimension(256,256);
  }

  public Dimension minimumSize(){
    return preferredSize();
  }

  /*
  int Width=-1, Height=-1;
  Image memImage = null;
  Graphics memGraphics = null;
  void checkImage(){
    if(size().width != Width || size().height != Height ){
      Width = size().width; Height = size().height;     
      memImage = this.createImage(Width,Height);
      memGraphics = memImage.getGraphics();
    }    
  }
*/

  Image memImage = null;
  Image[] memImages = null;

  //
  // work around some bug in Motif, 
  // which result in crushes of Netscape in Linux on stretched images  
  boolean workAroundNetscapeCrush = true;

  void makeImage(){
    
    int length = 256 >> colorModel.grid;
    
    int[] bytes = new int[length*length];
    if(workAroundNetscapeCrush){
      memImages = new Image[length]; 
    }
    int grid = colorModel.grid;
    for (int i = 0; i < length; i++){
      int offset = i*length;
      switch(colorModel.model){
      case ColorModel.R:
	int red = colorModel.color.getRed();
	for (int j = 0; j < length; j++)
	  bytes[j+offset] = 0xff000000 | (((red << 8)| (i<<grid))<<8 | (j << grid));
	break;
      case ColorModel.G: 
	int green = colorModel.color.getGreen();
	for (int j = 0; j < length; j++)
	  bytes[j+offset] = 0xff000000 | ((((i<<grid)<< 8)| green )<<8 | (j<<grid) );
	break;
      case ColorModel.B:
	int blue = colorModel.color.getBlue();
	for (int j = 0; j < length; j++)
	  bytes[j+offset] = 0xff000000 | ((((i<<grid)  << 8)| (j<<grid))<<8 | blue);
	break;
      case ColorModel.H:
	float h = colorModel.hsl[0];
	for (int j = 0; j < length; j++)
	  bytes[j+offset] = 0xff000000| Color.HSBtoRGB(h, (i<<grid)/255.0f, (j<<grid)/255.0f);
	break;
      case ColorModel.S:
	float s = colorModel.hsl[1];
	for (int j = 0; j < length; j++)
	  bytes[j+offset] = 0xff000000| Color.HSBtoRGB((i<<grid)/255.0f, s, (j<<grid)/255.0f);
	break;
      case ColorModel.L:
	float l = colorModel.hsl[2];      
	for (int j = 0; j < length; j++)
	  bytes[j+offset] = 0xff000000|
	    Color.HSBtoRGB((i<<grid)/255.0f, (j<<grid)/255.0f,l);
	break;
      }
      if(workAroundNetscapeCrush){
	memImages[i] = this.createImage(new MemoryImageSource(length,1,bytes,offset,length));
      }
    }
    if(!workAroundNetscapeCrush){
      memImage = this.createImage(new MemoryImageSource(length,length,bytes,0,length));
    }
  }

  public void update(Graphics g){
    paint(g);
  }

  public void paint(Graphics g) 
    {
      if(workAroundNetscapeCrush){
	for(int i=0;i < memImages.length;i++){
	  int y0 = (i*size().height)/memImages.length,
	  y1 = ((i+1)*size().height)/memImages.length;
	  if(y1-y0 > 0)
	    g.drawImage(memImages[i],0, y0, size().width, y1-y0, null);
	}
      } else {      
	g.drawImage(memImage,0,0,size().width,size().height,null);
      }
      
      int xcol=0,ycol=0;
      switch(colorModel.model){
      default:
      case ColorModel.R:
	ycol = colorModel.color.getGreen();
	xcol = colorModel.color.getBlue();
	break;
      case ColorModel.G:
	ycol = colorModel.color.getRed();
	xcol = colorModel.color.getBlue();
	break;
      case ColorModel.B:
	ycol = colorModel.color.getRed();
	xcol = colorModel.color.getGreen();
	break;
      case ColorModel.H:
	ycol = (int)(colorModel.hsl[1]*255);
	xcol = (int)(colorModel.hsl[2]*255);
	break;
      case ColorModel.S:
	ycol = (int)(colorModel.hsl[0]*255);
	xcol = (int)(colorModel.hsl[2]*255);
	break;
      case ColorModel.L:
	ycol = (int)(colorModel.hsl[0]*255);
	xcol = (int)(colorModel.hsl[1]*255);
	break;
      }

      xcol = (int)((xcol/255.0)*size().width);
      ycol = (int)((ycol/255.0)*size().height);


      //Draw color location
      g.setColor(Color.white);
      g.drawOval(xcol-2,ycol-2,5,5);
      
      g.setColor(Color.black);
      g.drawOval(xcol-3,ycol-3,7,7);
            
    }

  void makeColor(int x,int y){
    if(y < 0) y = 0;
    if(y > size().height-1) y =  size().height-1;
    if(x < 0) x = 0;
    if(x > size().width-1) x =  size().width-1;

    x = (x*255)/size().width;
    y = (y*255)/size().height;
    Color old = colorModel.color;
    switch(colorModel.model){
    case colorModel.R:
      colorModel.color = new Color(old.getRed(), y, x);break;
    case colorModel.G:
      colorModel.color = new Color(y, old.getGreen(), x);break;
    case colorModel.B:
      colorModel.color = new Color(y,x,old.getBlue());break;
    case colorModel.H:
      colorModel.color = Color.getHSBColor(colorModel.hsl[0], y/255.0f, x/255.0f);break;
    case colorModel.S:
      colorModel.color = Color.getHSBColor( y/255.0f,colorModel.hsl[1], x/255.0f);break;
    case colorModel.L:
      colorModel.color = Color.getHSBColor( y/255.0f, x/255.0f,colorModel.hsl[2]);break;
    }
    colorModel.makeHSL();
  }  

  public boolean mouseDrag(Event e, int x, int y){
    makeColor(x,y);
    parent.colorPanel.repaint();

    parent.colorSlider.makeImage();
    parent.colorSlider.repaint();

    return false;
  }
  public boolean mouseUp(Event e, int x, int y){
    makeColor(x,y);
    parent.colorPanel.repaint();
    parent.colorSlider.makeImage();
    parent.colorSlider.repaint();
    parent.initValues();
    repaint();
    return false;
  }
}

class ColorSlider extends Canvas {

  ColorModel colorModel;
  ColorChooser parent;  

  public ColorSlider(ColorModel colorModel, ColorChooser parent) {
    this.colorModel = colorModel;
    makeImage();    
    this.parent = parent;
  }

  public Dimension preferredSize(){
    return new Dimension(40,256);
  }

  public Dimension minimumSize(){
    return preferredSize();
  }

  Image memImage = null;
  void makeImage(){
    int length = (256 >> colorModel.grid);

    int bytes[] = new int[length];
    int c = 0, shift = 0;
    switch(colorModel.model){
    case ColorModel.R:
      c = 0xff000000|colorModel.color.getGreen()<<8 |colorModel.color.getBlue();
      shift = colorModel.grid + 16;
      for (int i = 0; i < length; i++)
	bytes[i] = (i<<shift) | c;
      break;
    case ColorModel.G:
      c = 0xff000000|colorModel.color.getRed()<<16 |colorModel.color.getBlue();
      shift = colorModel.grid + 8;
      for (int i = 0; i < length; i++)
	bytes[i] = (i<<shift) | c;
      break;
    case ColorModel.B:
      c = 0xff000000|colorModel.color.getRed()<<16 |colorModel.color.getGreen()<<8;
      shift = colorModel.grid;
      for (int i = 0; i < length; i++)
	bytes[i] = (i<<shift) |c;
      break;
    case ColorModel.H:
      for (int i = 0; i < length; i++)
	bytes[i] = Color.HSBtoRGB((i<<colorModel.grid)/255.0f,colorModel.hsl[1],colorModel.hsl[2]);
      break;
    case ColorModel.S:
      for (int i = 0; i < length; i++)
	bytes[i] = Color.HSBtoRGB(colorModel.hsl[0],(i<<colorModel.grid)/255.0f,colorModel.hsl[2]);
      break;
    case ColorModel.L:
      for (int i = 0; i < length; i++)
	bytes[i] = Color.HSBtoRGB(colorModel.hsl[0],colorModel.hsl[1],(i << colorModel.grid)/255.0f);
      break;
    }
    memImage = this.createImage(new MemoryImageSource(1,length,bytes,0,1));
  }

  public void update(Graphics g){
    paint(g);
  }

  void drawposition(Graphics g){
    int shift = size().width/4;
    g.setColor(getBackground());
    g.fillRect(0,0,shift,size().height);

    int level = 0;
    switch(colorModel.model){
    case colorModel.R:
      level = colorModel.color.getRed();break;
    case colorModel.G:
      level = colorModel.color.getGreen();break;
    case colorModel.B:
      level = colorModel.color.getBlue();break;
    case colorModel.H:
      level = (int)(colorModel.hsl[0]*255);break;
    case colorModel.S:
      level = (int)(colorModel.hsl[1]*255);break;
    case colorModel.L:
      level = (int)(colorModel.hsl[2]*255);break;
    }
    
    level = (int)((level/255.0)*size().height) - 1;      
    /*
       g.setColor(Color.white);
       g.fillRect(0,level-1,shift,3);
       g.setColor(Color.black);
       g.drawRect(0,level-2,shift-1,5);
       */
    int border = 2;
    int w = shift-1, h = 6;
    Color col1 = getBackground().darker();
    Color col2 = getBackground().brighter();
    g.setColor(col1);
    for(int i=0; i< border; i++) {
      g.drawLine(i,i+level,w-i,i+level);
      g.drawLine(i,i+level,i,h-i+level);
    }
    g.setColor(col2);
    for(int i=0; i<border; i++) {
      g.drawLine(w-i,h-i+level, w-i,i+level);
      g.drawLine(w-i,h-i+level, i,h-i+level);
    }
    
  }
  public void paint(Graphics g) 
    {
      int shift = size().width/4;

      //checkImage();      
      int border = 2;

      g.drawImage(memImage,shift+border, border, 
		  size().width-shift-2*border, size().height-2*border,null);
      drawposition(g);


      int w = size().width-1-shift, h = size().height-1;
      Color col1 = getBackground().darker();
      Color col2 = getBackground().brighter();
      g.setColor(col1);
      for(int i=0; i<border; i++) {
	g.drawLine(shift+i,i,shift+w-i,i);
	g.drawLine(shift+i,i,shift+i,h-i);
      }
      g.setColor(col2);
      for(int i=0; i<border; i++) {
	g.drawLine(shift+w-i,h-i, shift+w-i,i);
	g.drawLine(shift+w-i,h-i, shift+i,h-i);
      }

    }  

  void makeColor(int y){
    if(y < 0) y = 0;
    if(y > size().height-1) y =  size().height-1;
    int level = (y*255)/size().height;
    Color old = colorModel.color;
    switch(colorModel.model){
    case colorModel.R:
      colorModel.color = new Color(level, old.getGreen(), old.getBlue());break;
    case colorModel.G:
      colorModel.color = new Color(old.getRed(), level, old.getBlue());break;
    case colorModel.B:
      colorModel.color = new Color(old.getRed(), old.getGreen(),level);break;
    case colorModel.H:
      colorModel.color = Color.getHSBColor(level/255.0f,colorModel.hsl[1],colorModel.hsl[2]);break;
    case colorModel.S:
      colorModel.color = Color.getHSBColor(colorModel.hsl[0],level/255.0f,colorModel.hsl[2]);break;
    case colorModel.L:
      colorModel.color = Color.getHSBColor(colorModel.hsl[0],colorModel.hsl[1],level/255.0f);break;
    }
    colorModel.makeHSL();
  }  

  public boolean mouseUp(Event e, int x, int y){
    makeColor(y);
    parent.colorArea.makeImage();
    parent.colorArea.repaint();
    parent.colorPanel.repaint();
    parent.initValues();
    repaint();
    return false;
  }

  public boolean mouseDrag(Event e, int x, int y){
    makeColor(y);
    parent.colorPanel.repaint();
    drawposition(this.getGraphics());
    if(parent.alwaysRepaint){
      parent.colorArea.makeImage();
      parent.colorArea.repaint();
    }
    //repaint();
    return false;
  }
}

public class ColorChooser extends Panel {
  
  GridBagLayout layout = new GridBagLayout();

  //Choice formatChoice = new Choice();
  public static final int INT = 0, FLOAT = 1, HEX = 2;
  int formatValue = INT; // format of color representation

  TextField [] colorValue = new TextField[6];
  Checkbox[] cbColorModel = new Checkbox[6];
  String [] cbColorModelName = {"R:","G:","B:","H:","S:","L:"};
  CheckboxGroup cbgColorModel = new CheckboxGroup();

  ColorModel colorModel = new ColorModel(new Color(200,128,128),ColorModel.R,4);

  Checkbox[] cbGrid = new Checkbox[4];
  String [] cbGridName = {"x2","x4","x8","x16"};
  CheckboxGroup cbgGrid = new CheckboxGroup();

  ColorPanel colorPanel = null;
  ColorArea colorArea = null;
  ColorSlider colorSlider = null;

  boolean alwaysRepaint = true;  // do full repaint at each movie of mouse

  public ColorChooser(Color color) {

    colorModel.setColor(color);

    colorPanel = new ColorPanel(colorModel);
    colorArea = new ColorArea(colorModel,this);
    colorSlider = new ColorSlider(colorModel,this);

    this.setLayout(layout);

    Panel pColorArea1 = new BorderPanel(BorderPanel.RAISED); pColorArea1.setLayout(layout);
    Panel pColorArea = new BorderPanel(BorderPanel.RECESSED); pColorArea.setLayout(layout);
    WindowUtils.constrain(pColorArea1,pColorArea,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.,0,0,0,0);
    WindowUtils.constrain(this,pColorArea1,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.,0,0,0,0);
    WindowUtils.constrain(pColorArea,colorArea,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.,0,0,0,0);

    Panel pColorSlider = new BorderPanel(BorderPanel.RAISED); pColorSlider.setLayout(layout);
    WindowUtils.constrain(this,pColorSlider,1,0,1,1,GridBagConstraints.VERTICAL,
                          GridBagConstraints.CENTER,0.0,0.0,0,0,0,0);
    WindowUtils.constrain(pColorSlider,colorSlider,0,0,1,1,GridBagConstraints.VERTICAL,
                          GridBagConstraints.CENTER,1.,1.,0,0,0,0);

    Panel pControls = new Panel(); pControls.setLayout(layout);
    WindowUtils.constrain(this,pControls,2,0,1,1,GridBagConstraints.VERTICAL,
                          GridBagConstraints.CENTER,0.,0.,0,0,0,0);

    Panel pColor = new BorderPanel(BorderPanel.RAISED); pColor.setLayout(layout);
    Panel pColor1 = new BorderPanel(BorderPanel.RECESSED); pColor1.setLayout(layout);
    WindowUtils.constrain(pControls,pColor,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,0.5,0.5,0,0,0,0);
    WindowUtils.constrain(pColor,pColor1,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,0.5,0.5,0,0,0,0);

    WindowUtils.constrain(pColor1,colorPanel,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.,0,0,0,0);
    
       Panel pNumbers = new Panel(); pNumbers.setLayout(layout);

    /*       
       formatChoice.addItem("dec");formatChoice.addItem("hex");formatChoice.addItem("flt");
       formatChoice.select(formatValue);
       WindowUtils.constrain(pNumbers,new Label("Format:"),0,0,2,1,GridBagConstraints.NONE,
       GridBagConstraints.EAST,0.,0.,0,0,0,0);      
       WindowUtils.constrain(pNumbers,formatChoice,2,0,1,1,GridBagConstraints.NONE,
       GridBagConstraints.EAST,0.,0.,0,0,0,0);      
       */

    {
      for(int i=0;i < cbColorModel.length;i++){
	WindowUtils.constrain(pNumbers,new Label(cbColorModelName[i]),0,i,1,1,GridBagConstraints.NONE,
			      GridBagConstraints.EAST,0.,0.,0,0,0,0);      
	cbColorModel[i] = new Checkbox(null,cbgColorModel,false);
	WindowUtils.constrain(pNumbers,cbColorModel[i],1,i,1,1,GridBagConstraints.NONE,
			      GridBagConstraints.WEST,0.,0.,0,0,0,0);      
	colorValue[i] = new TextField(3);
	WindowUtils.constrain(pNumbers,colorValue[i],2,i,1,1,GridBagConstraints.NONE,
			      GridBagConstraints.WEST,0.,0.,0,0,0,0);      

      }
      cbgColorModel.setCurrent(cbColorModel[colorModel.model]);
    }
    WindowUtils.constrain(pControls,pNumbers,0,2,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.WEST,1.,1.,0,0,0,0);

 
    Panel pGrid = new Panel(); pGrid.setLayout(layout);
    WindowUtils.constrain(pControls,pGrid,0,3,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.,0,0,0,0);    
    {
      for(int i=0;i < cbGrid.length; i++){
	cbGrid[i] = new Checkbox(null,cbgGrid,false);
	WindowUtils.constrain(pGrid,cbGrid[i],i,0,1,1,GridBagConstraints.NONE,
			      GridBagConstraints.WEST,0.,0.,0,0,0,0);      
	WindowUtils.constrain(pGrid,new Label(cbGridName[i]),i,1,1,1,GridBagConstraints.NONE,
			      GridBagConstraints.WEST,0.,0.,0,0,0,0);      
	
      }
      cbgGrid.setCurrent(cbGrid[colorModel.grid-1]);
    }

    initValues();
  }

  static float ColorToFloat(int color){
    return ((int)(color/2.55f))/100.0f;
  }
  float round (float f){
    return ((int)(f*100+0.5))/100.0f;
  }

  void initValues(){
    switch(formatValue){
    case INT:
      colorValue[0].setText(Integer.toString(colorModel.color.getRed()));
      colorValue[1].setText(Integer.toString(colorModel.color.getGreen()));
      colorValue[2].setText(Integer.toString(colorModel.color.getBlue()));
      colorValue[3].setText(Integer.toString((int)(colorModel.hsl[0]*360)));
      colorValue[4].setText(Integer.toString((int)(colorModel.hsl[1]*100)));
      colorValue[5].setText(Integer.toString((int)(colorModel.hsl[2]*100)));
      break;
    case FLOAT:
      colorValue[0].setText(Float.toString(ColorToFloat(colorModel.color.getRed())));
      colorValue[1].setText(Float.toString(ColorToFloat(colorModel.color.getGreen())));
      colorValue[2].setText(Float.toString(ColorToFloat(colorModel.color.getBlue())));
      colorValue[3].setText(Float.toString(round(colorModel.hsl[0])));
      colorValue[4].setText(Float.toString(round(colorModel.hsl[1])));
      colorValue[5].setText(Float.toString(round(colorModel.hsl[2])));
      break;
    }
  }

  public boolean action(Event e, Object what){
    for(int i=0; i < cbColorModel.length;i++){
      if(e.target == cbColorModel[i]){
	colorModel.model = i;
	colorArea.makeImage();
	colorArea.repaint();
	colorSlider.makeImage();
	colorSlider.repaint();
	return true;
      }	
    }   
    for(int i=0; i < cbGrid.length;i++){
      if(e.target == cbGrid[i]){
	colorModel.grid = i+1;
	colorArea.makeImage();
	colorArea.repaint();
	colorSlider.makeImage();
	colorSlider.repaint();
	return true;
      }	      
    }
    Color old = colorModel.color;
    if(e.target == colorValue[0]){
      switch(formatValue){
      case INT:
	colorModel.color = new Color((Integer.valueOf(colorValue[0].getText())).intValue(),
				     old.getGreen(), old.getBlue());
	break;
      case FLOAT:
	colorModel.color = new Color((int)((Float.valueOf(colorValue[0].getText())).floatValue()*255),
				     old.getGreen(), old.getBlue());
	break;
      }	
    } else if(e.target == colorValue[1]){
      switch(formatValue){
      case INT:
	colorModel.color = new Color(old.getRed(), 
				   (Integer.valueOf(colorValue[1].getText())).intValue(),
				   old.getBlue());
	break;
      case FLOAT:
	colorModel.color = new Color(old.getRed(), 
				   (int)((Float.valueOf(colorValue[1].getText())).floatValue()*255),
				   old.getBlue());
	break;	
      }


    } else if(e.target == colorValue[2]){
      switch(formatValue){
      case INT:
	colorModel.color = new Color(old.getRed(), old.getGreen(),
				     (Integer.valueOf(colorValue[2].getText())).intValue());
	break;
      case FLOAT:
	colorModel.color = new Color(old.getRed(), old.getGreen(),
				     (int)((Float.valueOf(colorValue[2].getText())).floatValue()*255));
	break;
      }
	
    } else if(e.target == colorValue[3]){
      switch(formatValue){
      case INT:
	colorModel.color = Color.getHSBColor((Integer.valueOf(colorValue[3].getText())).intValue()/360.0f,
					   colorModel.hsl[1],colorModel.hsl[2]);
	break;
      case FLOAT:
	colorModel.color = Color.getHSBColor((Float.valueOf(colorValue[3].getText())).floatValue(),
					   colorModel.hsl[1],colorModel.hsl[2]);
	break;
      }

    } else if(e.target == colorValue[4]){
      switch(formatValue){
      case INT:
	colorModel.color = Color.getHSBColor(colorModel.hsl[0],
					   (Integer.valueOf(colorValue[4].getText())).intValue()/100.0f,
					   colorModel.hsl[2]);
	break;
      case FLOAT:
	colorModel.color = Color.getHSBColor(colorModel.hsl[0],
					   (Float.valueOf(colorValue[4].getText())).floatValue(),
					   colorModel.hsl[2]);
	break;
      }
    } else if(e.target == colorValue[5]){
      switch(formatValue){
      case INT:
	colorModel.color = Color.getHSBColor(colorModel.hsl[0],colorModel.hsl[1],
					   (Integer.valueOf(colorValue[5].getText())).intValue()/100.0f);
	break;
      case FLOAT:
	colorModel.color = Color.getHSBColor(colorModel.hsl[0],colorModel.hsl[1],
					   (Float.valueOf(colorValue[5].getText())).floatValue());
	break;
      }
    }         
    colorModel.makeHSL();
    initValues();colorPanel.repaint();
    colorArea.makeImage(); colorArea.repaint();
    colorSlider.makeImage();colorSlider.repaint();
    return true;

  }
  
  public String getAppletInfo() {
    return getClass().getName() + 
      " - Copyright (C) 1996 by Vladimir Bulatov <V.Bulatov.ic.ac.uk>.  All rights reserved.";
  }

  public static void main(String[] args) {
    ColorChooser applet = new ColorChooser(new Color(124,125,192));
    Frame frame = new AppletFrame("ColorChooser Test");
    frame.add("Center", applet);
    frame.resize(450, 350);
    frame.show();
  }
}
