package PVS.POVTextureEditor;
// TextureEditor.java - a POVray  Texture editor
//
// Copyright (C) 1996 by Vladimir Bulatov <V.Bulatov@ic.ac.uk>.  
//            All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.


import java.awt.*;
import java.io.*;
import java.util.Vector;

import PVS.Utils.*;

public class TextureEditor extends Panel {
  
  GridBagLayout layout = new GridBagLayout();

  Button addButton = new Button("Add");
  Button removeButton = new Button("Remove");
  Button upButton = new Button("Up");
  Button downButton = new Button("Down");

  Button printButton = new Button("Print");
  Button copyButton = new Button("Copy");
  Button pasteButton = new Button("Paste");
  Button updateButton = new Button("Update");
  Button undoButton = new Button("Undo");


  List layersList = new List(5,false); 
  SamplePanel samplePanel = new SamplePanel();


  POVTextureLayer layer = null;
  POVTexture texture = null;
  POVDeclare declare = null;


  TabbedPanel panelAll;

  PigmentEditor pigmentEditor = new PigmentEditor(new POVPigment());
  NormalEditor normalEditor = new NormalEditor(new POVNormal());
  FinishEditor finishEditor = new FinishEditor(new POVFinish());

  List undoList =  new List();


  public TextureEditor(POVTexture texture) {

    this.texture = texture;
    initLayersList();

    this.setLayout(layout);

    panelAll = new TabbedPanel(this);
    panelAll.addItem("Pigment",pigmentEditor);
    panelAll.addItem("Finish",finishEditor);
    panelAll.addItem("Normal",normalEditor);

    BorderPanel panel1 = new BorderPanel(); panel1.setLayout(layout);
    BorderPanel panel2 = new BorderPanel(); panel2.setLayout(layout);
    //BorderPanel panel3 = new BorderPanel(); panel3.setLayout(layout);
    Panel panel4 = new Panel(); panel4.setLayout(layout);
    Panel panel5 = new Panel(); panel5.setLayout(layout);

    WindowUtils.constrain(panel4,upButton,0,0,1,1,
			  GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1.);    
    WindowUtils.constrain(panel4,downButton,0,1,1,1,
			  GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1.);
    WindowUtils.constrain(panel4,addButton,0,2,1,1,
			  GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1.);    
    WindowUtils.constrain(panel4,removeButton,0,3,1,1,
			  GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1.);


    WindowUtils.constrain(panel5,printButton,0,0,1,1,GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1.);
    WindowUtils.constrain(panel5,copyButton,0,1,1,1,GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1.);
    WindowUtils.constrain(panel5,pasteButton,0,2,1,1,GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1.);
    WindowUtils.constrain(panel5,updateButton,0,3,1,1,GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1.,0,5,0,0);
    WindowUtils.constrain(panel5,undoButton,0,4,1,1,GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1.);
    /*
    WindowUtils.constrain(panel3,undoList,0,0,1,1,GridBagConstraints.VERTICAL,
                          GridBagConstraints.CENTER,1.,1.);
    WindowUtils.constrain(panel3,panel5,1,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);
    */


    WindowUtils.constrain(panel1,layersList,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,0.2);    
    WindowUtils.constrain(panel1,panel4,1,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,0.2);    

    WindowUtils.constrain(panel1,undoList,0,1,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,0.1,0.2);    
    WindowUtils.constrain(panel1,panel5,1,1,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,0.1,0.2);    

    WindowUtils.constrain(panel1,samplePanel,0,2,2,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);    

    WindowUtils.constrain(panel2,panelAll,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);    

    WindowUtils.constrain(this,panel1,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);    
    WindowUtils.constrain(this,panel2,1,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);    
  }
  
  void initLayersList(){
    if(layersList.countItems()>0)
      layersList.clear();
    if(texture != null){
      for(int i = 0;i < texture.layers.size();i++){
	layersList.addItem("layer"+Integer.toString(i));
      }    
      if(texture.layers.size() > 0){
	layer = (POVTextureLayer)texture.layers.elementAt(0);
	layersList.select(0);
      }
      else 
	layer = new POVTextureLayer(); // to avoid problems
    }
  }

  public boolean action(Event e, Object what){   
    
    if(e.target == layersList){
      for(int i = 0;i < texture.layers.size();i++){
	if(what.equals("layer"+Integer.toString(i))){
	  layer = (POVTextureLayer)texture.layers.elementAt(i);
	  break;
	}
      }
      this.init(layer);
      return true;
    } else if(e.target == updateButton){
      doUpdate();
    } else if(e.target == undoButton){
      doUndo();
    } else if(e.target == pasteButton){
      doPaste();
    } else if(e.target == upButton){
      doUp();
    } else if(e.target == downButton){
      doDown();
    } else if(e.target == copyButton){
      doCopy();
    } else if(e.target == printButton){
      doPrint();
    } else if(e.target == addButton){
      doAdd();
    } else if(e.target == removeButton){
      doRemove();
    }    
    return false;
  }

  void doUp(){
    int ind = layersList.getSelectedIndex();
    int ind1 = (ind > 0) ? ind-1 : 0;
    layersList.select(ind1);
    if(ind != ind1){
      POVTextureLayer l = (POVTextureLayer)texture.layers.elementAt(ind);
      texture.layers.removeElementAt(ind);
      texture.layers.insertElementAt(l,ind1);	
    }    
  }

  void doDown(){
    int ind = layersList.getSelectedIndex();
    int ind1 = (ind < layersList.countItems()-1)?
      ind+1:layersList.countItems()-1;
    layersList.select(ind1);
    if(ind != ind1){
      POVTextureLayer l = (POVTextureLayer)texture.layers.elementAt(ind);
      texture.layers.removeElementAt(ind);
      texture.layers.insertElementAt(l,ind1);	
    }
  }
  
  void doAdd(){
    layer = new POVTextureLayer();
    texture.addLayer(layer);
    int n = layersList.countItems();
    layersList.addItem("layer"+n);
    layersList.select(n);
    init(layer);
  }

  void doRemove(){
    int[] ind = layersList.getSelectedIndexes();
    for(int i=ind.length-1; i >=0; i--)
      texture.layers.removeElementAt(ind[i]);
    initLayersList();
    if(layersList.countItems() > 0){
      int n = ind[0]-1;
      if(n<0)
	n = 0;
      layersList.select(n);
      layer = (POVTextureLayer)texture.layers.elementAt(n);
      init(layer);
    }
  }

  void doPaste(){
    POVTexture tex = (POVTexture)Clipboard.get("POVTexture");
    if(tex != null){
      int n = layersList.countItems();
      for(int i=0; i < tex.layers.size(); i++){
	texture.addLayer((POVTextureLayer)((POVTextureLayer)tex.layers.
					   elementAt(i)).clone());
	layersList.addItem("layer"+(i+n));
      }
      layer = (POVTextureLayer)((POVTextureLayer)tex.layers.
					   elementAt(0)).clone();
      init(layer);
      layersList.select(n);
      return;
    }
    POVPigment pigment = (POVPigment)Clipboard.get("POVPigment");
    if(pigment != null){
      if(layer != null){
	layer.pigment = (POVPigment)pigment.clone();
	init(layer);
      }
      return;
    }
    POVColorMap color_map = (POVColorMap)Clipboard.get("POVColorMap");
    if(color_map != null){
      if(layer != null){
	layer.pigment.color_map = (POVColorMap)color_map.clone();
	init(layer);
      }
      return;
    }
    POVColor color = (POVColor)Clipboard.get("POVColor");
    if(color != null){
      if(layer != null){
	layer.pigment.color = (POVColor)color.clone();
	init(layer);
      }
      return;
    }
    
  }

  void doCopy(){
    if(layer != null){
      POVTexture tex = new POVTexture();
      tex.addLayer(layer);
      Clipboard.put(tex);
    }
  }

  void doPrint(){
    Frame frame = new DestroyableFrame("Texture output");
    TextArea text = new TextArea(texture.toString());
    text.setFont(new Font("Dialog",Font.PLAIN,14));
    
    frame.add("Center",text);
    frame.pack(); frame.show();
  }

  Vector undoArray = new Vector();
  int currentElement = 0;
  int startShift = 0;

  void doUndo(){
    WindowUtils.getMainWindow(this).setCursor(Frame.WAIT_CURSOR);
    // if there is selection, we will undo to this selection
    int selected = undoList.getSelectedIndex();
    if(selected != currentElement){
      undoList.replaceItem("   "+declare.name+"."+(currentElement+startShift),
			   currentElement);
      undoList.replaceItem("->"+declare.name+"."+(selected+startShift),
			   selected);
    }
    texture = (POVTexture)undoArray.elementAt(selected);
    samplePanel.setDrawable(texture); 
    if(declare != null)    // store this texture  
      declare.o = texture;
    texture = (POVTexture)texture.clone(); // we will edit the copy

    // let's remember which layer was selected before undo
    int index = layersList.getSelectedIndex(); 
    initLayersList();
    if(index < 0 ){
      index = 0; 
      layersList.select(0);
    }
    layer = (POVTextureLayer)texture.layers.elementAt(index);
    init(layer);


    undoList.select(selected);
    currentElement = selected;

    WindowUtils.getMainWindow(this).setCursor(Frame.DEFAULT_CURSOR);
  }

  void doUpdate(){
    WindowUtils.getMainWindow(this).setCursor(Frame.WAIT_CURSOR);

    // clean the tail 
    int nsize = undoArray.size();
    if(currentElement < nsize){
      undoList.delItems(currentElement+1,nsize-1);      
      for(int i = currentElement+1;i < nsize; i++){
	undoArray.removeElementAt(currentElement+1);
      }
    }

    undoList.replaceItem("   "+declare.name+"."+(currentElement+startShift),
			 currentElement);
    currentElement++;
    undoList.addItem("->"+declare.name+"."+(currentElement+startShift));
    undoList.select(currentElement);

    try{
      texture.makeThumbnail(samplePanel);    
    } catch(Exception e){
      e.printStackTrace();
    }
    samplePanel.setDrawable(texture);
    undoArray.addElement(texture);

    if(undoArray.size() > Options.undoLevel.value){
      // to prevent overflowing of undo buffer
      startShift++;
      currentElement--;
      undoArray.removeElementAt(0);
      undoList.delItem(0);
    }
    if(declare!=null)    // store this texture  
      declare.o = texture;
    
    texture = (POVTexture)texture.clone(); // new texture for editing
    int index = layersList.getSelectedIndex();
    if(index < 0 ){
      index = 0; 
      layersList.select(0);
    }
    layer = (POVTextureLayer)texture.layers.elementAt(index);
    init(layer);

    undoButton.enable();
    WindowUtils.getMainWindow(this).setCursor(Frame.DEFAULT_CURSOR);
  }


  public boolean handleEvent(Event e){   
    if(e.target == layersList){
      switch(e.id){
      case Event.LIST_SELECT:
	layer = (POVTextureLayer)texture.
	  layers.elementAt(((Integer)e.arg).intValue());
	this.init(layer);
	//
	break;
      case Event.LIST_DESELECT:
	//samplePanel.unsetDrawable();
	break;
      }
    } else  if(e.target == undoList){
      switch(e.id){
      case Event.LIST_SELECT:
	int selected = ((Integer)e.arg).intValue();
	samplePanel.setDrawable((Drawable)undoArray.elementAt(selected));
	undoButton.enable();
	return true;
      case Event.LIST_DESELECT:
	undoButton.disable();
	return true;
      case Event.ACTION_EVENT:
	System.out.println("undoList:action");	
      }
      return true;
    }

    return super.handleEvent(e);
  }

  
  void storeBack(){
    if(declare != null){ // if there was old 
      ((POVTexture)(declare.o)).assign((POVTexture)undoArray.
				     elementAt(currentElement));
    }
  }
  

  public void init(POVDeclare declare){
    //storeBack();
    this.declare = declare;
    Frame f = WindowUtils.getMainWindow(this);
    if(f != null){
      f.setTitle("Texture: "+declare.name);
      f.show();
    }

    undoArray.removeAllElements();
    // store new texture to be able undo later
    undoArray.addElement((POVTexture)declare.o); 
    samplePanel.setDrawable((POVTexture)declare.o);

    texture = (POVTexture)((POVTexture)declare.o).clone();
    currentElement = 0; // it starts from 0
    startShift = 0;     // initial shift in undo buffer

    initLayersList();
    init(layer);

    if(undoList.countItems() > 0) // Motif complains if list is empty
      undoList.clear();
    undoList.addItem("->"+declare.name+"."+currentElement);
    undoList.select(currentElement);
    
    undoButton.enable();

  }

  public void init(POVTexture texture){    
    this.texture = texture;    
    initLayersList();
    init(layer);
    samplePanel.setDrawable(texture);
  } 

  public void init(POVTextureLayer layer){
    this.layer = layer;
    pigmentEditor.init(layer.pigment);
    normalEditor.init(layer.normal);
    finishEditor.init(layer.finish);
  } 

  public static void main(String[] args) throws IOException{
    POVTexture texture = new POVTexture();
    texture.addLayer(new POVTextureLayer());
    texture.addLayer(new POVTextureLayer());
    texture.addLayer(new POVTextureLayer());

    TextureEditor editor = new TextureEditor(texture);
    Frame frame = new AppletFrame("Texture Editor");
    frame.add("Center", editor);

    editor.init(new POVDeclare("My texture",texture));
    frame.pack();
    frame.show();
  }
}
