package PVS.POVTextureEditor;
// PigmentEditorFrame.java - a POVTextureEditor
//
// Copyright (C) 1996 by Vladimir Bulatov <V.Bulatov@ic.ac.uk>.  
//            All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.


import java.awt.*;
import java.io.*;
import java.util.Vector;

import PVS.Utils.*;

public class PigmentEditorFrame extends Panel {
  
  GridBagLayout layout = new GridBagLayout();

  Button undoButton = new Button("Undo");
  //Button redoButton = new Button("Redo");
  Button copyButton = new Button("Copy");
  Button pasteButton = new Button("Paste");
  Button printButton = new Button("Print");
  Button updateButton = new Button("Update");

  List undoList;

  SamplePanel samplePanel = new SamplePanel();

  POVPigment pigment = null; // current pigment
  POVDeclare declare = null; // what we are editing now

  PigmentEditor pigmentEditor = new PigmentEditor(new POVPigment());

  public PigmentEditorFrame(POVPigment pigment) {

    undoList = new List(); // list to show current undo's

    this.setLayout(layout);
    
    this.init(pigment);

    BorderPanel panel1 = new BorderPanel(); panel1.setLayout(layout);
    BorderPanel panel2 = new BorderPanel(); panel2.setLayout(layout);
    Panel panel3 = new Panel(); panel3.setLayout(layout);

    WindowUtils.constrain(panel3,copyButton,0,0,1,1,
			  GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1);    
    WindowUtils.constrain(panel3,pasteButton,0,1,1,1,
			  GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1);    
    WindowUtils.constrain(panel3,printButton,0,2,1,1,
			  GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1);    
    WindowUtils.constrain(panel3,updateButton,0,3,1,1,
			  GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1);    
    WindowUtils.constrain(panel3,undoButton,0,4,1,1,
			  GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,1);    
    /*
    WindowUtils.constrain(panel3,redoButton,0,5,1,1, 
			  GridBagConstraints.HORIZONTAL, 
			  GridBagConstraints.CENTER,1.,1);    
			  */
    WindowUtils.constrain(panel2,undoList,0,0,1,1,GridBagConstraints.VERTICAL,
                          GridBagConstraints.CENTER,0.1,1);    
    WindowUtils.constrain(panel2,panel3,1,0,1,1,GridBagConstraints.VERTICAL,
                          GridBagConstraints.CENTER,1.,1);    
    WindowUtils.constrain(panel1,panel2,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,0.2);    
    WindowUtils.constrain(panel1,samplePanel,0,1,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);    

    WindowUtils.constrain(this,panel1,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);    
    WindowUtils.constrain(this,pigmentEditor,1,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);    
    undoButton.disable();
    //redoButton.disable();
  }
  

  public boolean action(Event e, Object what){   
    
    if(e.target == updateButton){
      doUpdate(); return true;
    }else if(e.target == undoButton){
      doUndo(); return true;
    //}else if(e.target == redoButton){
    //  doRedo(); return true;
    }else if(e.target == pasteButton){
      doPaste(); return true;
    }else if(e.target == copyButton){
      doCopy(); return true;
    }else if(e.target == printButton){
      doPrint(); return true;
    }
    return false;
  }

  void doCopy(){
    Clipboard.put(pigment.clone());
  }

  void doPaste(){
    POVPigment pigm = (POVPigment)Clipboard.get("POVPigment");
    if(pigm != null){
      pigment = pigm;
      pigmentEditor.init(pigment);
      return;
    }
  }

  Vector undoArray = new Vector();
  int currentElement = 0;
  int startShift = 0;

  void doUndo(){
    WindowUtils.getMainWindow(this).setCursor(Frame.WAIT_CURSOR);
    // if there is selection, we will undo to this selection
    int selected = undoList.getSelectedIndex();
    if(selected != currentElement){
      undoList.replaceItem("   "+declare.name+"."+(currentElement+startShift),
			   currentElement);
      undoList.replaceItem("->"+declare.name+"."+(selected+startShift),
			   selected);
    }
    pigment = (POVPigment)undoArray.elementAt(selected);
    samplePanel.setDrawable(pigment); 
    if(declare!=null)    // store this pigment
      declare.o = pigment;
    pigment = (POVPigment)pigment.clone(); // we will edit the copy
    pigmentEditor.init((POVPigment)pigment);
    undoList.select(selected);
    currentElement = selected;

    WindowUtils.getMainWindow(this).setCursor(Frame.DEFAULT_CURSOR);
  }

  /**
    response to update button
   */
  void doUpdate(){
    WindowUtils.getMainWindow(this).setCursor(Frame.WAIT_CURSOR);
    
    // clean the tail 
    int nsize = undoArray.size();
    if(currentElement < nsize - 1){
      undoList.delItems(currentElement+1,nsize-1);
      for(int i = currentElement+1;i < nsize; i++){
	undoArray.removeElementAt(currentElement+1);
      }
    }

    undoList.replaceItem("   "+declare.name+"."+(currentElement+startShift),
			 currentElement);
    currentElement++;
    undoList.addItem("->"+declare.name+"."+(currentElement+startShift));
    undoList.select(currentElement);

    try{
      pigment.makeThumbnail(samplePanel);    
    } catch(Exception e){
      e.printStackTrace();
    }
    samplePanel.setDrawable(pigment);
    undoArray.addElement(pigment);

    if(undoArray.size() > Options.undoLevel.value){
      // to prevent overflowing of undo buffer
      startShift++;
      currentElement--;
      undoArray.removeElementAt(0);
      undoList.delItem(0);
    }
    if(declare!=null)    // store this pigment
      declare.o = pigment;
    
    pigment = (POVPigment)pigment.clone();
    pigmentEditor.init(pigment);

    undoButton.enable();
    //redoButton.disable();

    WindowUtils.getMainWindow(this).setCursor(Frame.DEFAULT_CURSOR);
  }
  
  void doPrint(){
    Frame frame = new DestroyableFrame("Pigment output");
    TextArea text = new TextArea(pigment.toString());
    text.setFont(new Font("Dialog",Font.PLAIN,14));
    
    frame.add("Center",text);
    frame.pack(); frame.show();
  }

  public boolean handleEvent(Event e){   
    if(e.target == undoList){
      switch(e.id){
      case Event.LIST_SELECT:
	int selected = ((Integer)e.arg).intValue();
	samplePanel.setDrawable((POVPigment)undoArray.elementAt(selected));
	undoButton.enable();
	return true;
      case Event.LIST_DESELECT:
	undoButton.disable();
	return true;
      case Event.ACTION_EVENT:
	System.out.println("List:action");	
      }
      return true;
    }
    return super.handleEvent(e);
  }

  /**
    used to start edit another pigment
   */
  public void init(POVDeclare declare){
    this.declare = declare;
    Frame f = WindowUtils.getMainWindow(this);
    if(f != null){
      f.setTitle("Pigment: "+declare.name);
      f.show();
    }

    undoArray.removeAllElements();
    // store pigment to be able undo later
    undoArray.addElement((POVPigment)declare.o); 
    samplePanel.setDrawable((POVPigment)declare.o);

    // we will edit the copy
    pigment = (POVPigment)((POVPigment)declare.o).clone();
    currentElement = 0; // it starts from 0
    startShift = 0;     // initial shift in undo buffer

    pigmentEditor.init(pigment); // currently edited pigment

    if(undoList.countItems() > 0) // Motif complains if list is empty
      undoList.clear();
    undoList.addItem("->"+declare.name+"."+currentElement);
    undoList.select(currentElement);
    
    undoButton.enable();

  }

  public void init(POVPigment pigment){
    this.pigment = pigment;
    pigmentEditor.init(pigment);
    samplePanel.setDrawable(pigment);
  } 

  public static void main(String[] args) throws IOException{
    POVPigment pigment = new POVPigment();

    PigmentEditorFrame editor = new PigmentEditorFrame(pigment);
    Frame frame = new AppletFrame("Pigment Editor");
    frame.add("Center", editor);
    frame.pack();
    frame.show();
    editor.init(new POVDeclare("My Pigment",new POVPigment()));
  }
}

