package PVS.POVTextureEditor;
// POVThumbnailObject.java - a POVray  Texture editor
//
// Copyright (C) 1996 by Vladimir Bulatov <V.Bulatov@ic.ac.uk>.  
//            All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.


import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.util.Observer;
import java.util.Observable;
import java.awt.Image;
import java.awt.Graphics;
import java.awt.Color;
import java.awt.FontMetrics;

class POVThumbnailObject extends Object implements Drawable, Observer{

  String fullFileName = null;
  Observer observer; // who to inform, when we have finished
  boolean waitingForImage = false;
  Image thumbnailImage = null;
  boolean isModified = false;
  // should be defined by derived class
  String povFile = "texture.pov";
  String incFile = "texture.inc";
  static String renderedFile = "temp.tga";


  void setFileName(String name,String directory){
    if(directory == null){
      directory = System.getProperty("user.dir");
    }
    fullFileName = new String(directory +File.separator+ name + ".tga");    
  }

  void rename(String newname){
    File f = new File(fullFileName);
    String path = f.getParent();
    fullFileName = path+newname+".tga";
    System.out.println("renamed to "+fullFileName);
  }

  void makeThumbnail(Observer observer) throws IOException {
    waitingForImage = true;
    thumbnailImage = null;  // old image is worthless

    this.observer = observer;
    String[] command = new String[4];
    command[0] = System.getProperties().getProperty("povray.path");
    String home = System.getProperties().getProperty("home");
    if(home == null)
      home = ".";

    if(command[0] == null){ // Hmm..
      command[0] = "povray";
    }

    Defaults.CheckFile(home+File.separator+"povray.def");
    Defaults.CheckFile(home+File.separator+povFile);
    command[1] = home+"/povray.def";
    command[2] = "-I"+home+File.separator+povFile;
    command[3] = "-O"+renderedFile;
    //command[3] = "-O"+fullFileName+"$";
    POVRenderingQueue.submit(this,command, incFile,this.toString());
  }

  public void draw(Graphics g, int width, int height){
    if(thumbnailImage == null  && !waitingForImage){
      try{
	thumbnailImage = ImageTGA.read(new FileInputStream(fullFileName));
      } catch(Exception e){
	thumbnailImage = null;
      }
    }
    if(thumbnailImage != null){
      int imw = thumbnailImage.getWidth(null);
      int imh = thumbnailImage.getHeight(null);
      int w1 = imw*height/imh;
      if(w1 < width){  // place is wider than picture
	g.setColor(Color.lightGray);
	g.fillRect(0,0,(width-w1)/2,height); 	  
	g.fillRect(width -(width-w1)/2, 0,(width-w1)/2,height); 	  
	g.drawImage(thumbnailImage,(width-w1)/2,0,w1,height,null); 
      } else {  // place is higher than picture
	int h1 = imh*width/imw;
	g.setColor(Color.lightGray);
	g.fillRect(0,0,width,(height-h1)/2); 	  
	g.fillRect(0,height-(height-h1)/2-1,width,(height-h1)/2); 	  
	g.drawImage(thumbnailImage,0,(height-h1)/2,width,h1,null); 	  
      }	
    } else {
      g.setColor(Color.white);
      g.fillRect(0,0,width,height);
      g.setColor(Color.black);
      g.drawRect(0,0,width-1,height-1);

      FontMetrics fm = g.getFontMetrics();
      int h = fm.getHeight()+fm.getMaxDecent();
      String notrendered = "not rendered";
      int w = fm.stringWidth(notrendered);
      g.drawString(notrendered,(width -w)/2,(height - fm.getMaxDecent())/2);
    }
  }

  /**
    this method is called by renderer, when new rendered picture is ready
   */
  public void update(Observable o, Object arg){
    waitingForImage = false;
    //System.out.println("ready: "+fullFileName);
    try {
      //thumbnailImage = ImageTGA.read(new FileInputStream(fullFileName+"$"));
      thumbnailImage = ImageTGA.read(new FileInputStream(renderedFile));
      if(observer != null)
	observer.update(null,this);
      else{         // special case of doUpdateAll
	saveImage();  // we will save picture and release memory
	thumbnailImage = null;
      }	
      isModified = true;
    } catch(Exception ex){
      ex.printStackTrace(System.out);
    }    
  }

  public void saveImage(){
    try {
      if(thumbnailImage != null){
	File f = new File(fullFileName);
	POVParse.makeBackup(f);
	OutputStream out = new 
	  BufferedOutputStream(new FileOutputStream(fullFileName));
	ImageTGA.write(thumbnailImage,out);
	out.close();
      }
    } catch(Exception ex){
      ex.printStackTrace(System.out);
    }
  }
}
