package PVS.POVTextureEditor;
// ParsePOV.java - a POV file ray parser
//
// Copyright (C) 1996 by Vladimir Bulatov <V.Bulatov@ic.ac.uk>.  
//                  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.


import java.awt.*;
import java.awt.image.*;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Vector;
import java.io.*;

import PVS.Utils.*;

/**
 * A class to turn an input POV ray file into a set ob POV objects.
 * @version 1.1, 14 May 1996
 * @author  Vladimir Bulatov
 */
public class POVParse {

  /**
    test if next token is what we are expecting
   */
   public static int 
    expect(POVStreamTokenizer tokenizer, int what, String message) throws IOException {
      int token = tokenizer.nextToken();
      if(token!=what){
	switch(tokenizer.ttype){
	default:
	  throw new IOException(currentFile.getName() + ":"+ tokenizer.lineno()+message+
				" got \'"+(char)tokenizer.ttype+"\'");    
	case StreamTokenizer.TT_NUMBER:
	  throw new IOException(currentFile.getName() + ":"+ tokenizer.lineno()+message+
				" got \""+tokenizer.nval+"\"");    
	case StreamTokenizer.TT_WORD:
	  throw new IOException(currentFile.getName() + ":"+ tokenizer.lineno()+message+
				" got \""+tokenizer.sval+"\"");    	  
	}
      }
      return token;
    }

   public static int 
    expect(POVStreamTokenizer tokenizer, int what) throws IOException {
      switch(what){
      default:
	return expect(tokenizer,what," \""+(char)what+"\""+ " expected");
      case StreamTokenizer.TT_NUMBER:
	return expect(tokenizer,what," number expected");
      case StreamTokenizer.TT_WORD:
	return expect(tokenizer,what," word expected");
      }
    }

   public static int 
    expect(POVStreamTokenizer tokenizer, String what, String message) 
      throws IOException {
	int token = tokenizer.nextToken();	
	if(token !=StreamTokenizer.TT_WORD || !tokenizer.sval.equals(what))
	  throw new IOException(currentFile.getName() + ":"+ 
				tokenizer.lineno()+message);    
	return token;
      }

   public static int 
     expect(POVStreamTokenizer tokenizer, String what) throws IOException{
       return expect(tokenizer,what, " " + what + " expected"); 
     }

   public static void exception(POVStreamTokenizer tokenizer,String message) 
     throws IOException{
      switch(tokenizer.ttype){
      default:
	throw new IOException(currentFile.getName() + ":"+ tokenizer.lineno()+" " 
			      + message +
			      " got "+" \""+(char)tokenizer.ttype+"\"");    
      case StreamTokenizer.TT_EOF:
	throw new IOException(currentFile.getName() + ":"+ tokenizer.lineno()+" " 
			      + message + " got end of file");    
      case StreamTokenizer.TT_NUMBER:
	throw new IOException(currentFile.getName() + ":"+ tokenizer.lineno()+" " 
			      + message + " got "+tokenizer.nval);    
      case StreamTokenizer.TT_WORD:
	throw new IOException(currentFile.getName() + ":"+ tokenizer.lineno()+" " 
			      + message + " got "+tokenizer.sval);           
      }
   }

   public static POVNumber 
     getNumber(POVStreamTokenizer tokenizer)throws IOException {
       switch(tokenizer.nextToken()){
       case StreamTokenizer.TT_NUMBER:
	 return new POVNumber(tokenizer.nval);
       case StreamTokenizer.TT_WORD:
	 POVNumber num = findNumber(tokenizer.sval);
	 if(num != null){
	   return num;
	 } else {
	   exception(tokenizer,"number expected");
	 } break;
       default:
	 exception(tokenizer,"number expected");
	 break;
       }
       return null;
     }

   public static DoubleR [] 
     getVector(POVStreamTokenizer tokenizer,int n)throws IOException {
       DoubleR [] c = new DoubleR[n];
       for(int i=0;i<n;i++)
	 c[i] = new DoubleR(0.);
       switch(tokenizer.nextToken()){
       case '<':
	 for(int i=0;i<n;i++){
	   if(tokenizer.nextToken()!=StreamTokenizer.TT_NUMBER)
	     throw new IOException(currentFile.getName() + ":"+ tokenizer.lineno()+" float expected");
	   c[i].value = tokenizer.nval;
	   if(i < n-1){
	     if(tokenizer.nextToken()!=',')
	       exception(tokenizer," \",\" expected");
	   } else {
	     if(tokenizer.nextToken()!='>')
	       exception(tokenizer," \">\" expected");
	   }
	 }
	 break;
       case StreamTokenizer.TT_NUMBER:
	 for(int i=0;i<n;i++)
	   c[i].value = tokenizer.nval;
	 break;
       case StreamTokenizer.TT_WORD:
	 if(tokenizer.sval.equals("x")){
	   c[0].value = 1.; c[1].value = c[2].value = 0;
	 } else if(tokenizer.sval.equals("y")){
	   c[1].value = 1.; c[0].value = c[2].value = 0;
	 } else if(tokenizer.sval.equals("z")){
	   c[2].value = 1.; c[0].value = c[1].value = 0;
	 } else {
	   exception(tokenizer, " vector expected");
	 } 
	 break;
	 
       }
       return c;
     }

   public static DoubleR [] 
     getDirection(POVStreamTokenizer tokenizer)throws IOException {
       return getVector(tokenizer,3);
     }

   
   public static POVColor getColor(POVStreamTokenizer tokenizer)throws IOException {

     POVColor color = new POVColor();     
     tokenizer.nextToken();
     //expect(tokenizer,"color");
     if(tokenizer.ttype != StreamTokenizer.TT_WORD ||
	!(tokenizer.sval.equals("color") || 
	  tokenizer.sval.equals("colour")))
       tokenizer.pushBack();
     POVColor clr;
     while(true){
       if(tokenizer.nextToken() == StreamTokenizer.TT_WORD){
	 if(tokenizer.sval.equals("rgbf")){
	   DoubleR[] c = getVector(tokenizer,4);
	   return new POVColor(c[0].value,c[1].value,c[2].value,c[3].value);
	 } else if(tokenizer.sval.equals("rgb")){
	   DoubleR[] c = getVector(tokenizer,3);
	   return new POVColor(c[0].value,c[1].value,c[2].value,0.0);      
	 } else if(tokenizer.sval.equals("red")){
	   expect(tokenizer,StreamTokenizer.TT_NUMBER);
	   color.r = tokenizer.nval;
	 } else if(tokenizer.sval.equals("green")){
	   expect(tokenizer,StreamTokenizer.TT_NUMBER);
	   color.g = tokenizer.nval;
	 } else if(tokenizer.sval.equals("blue")){
	   expect(tokenizer,StreamTokenizer.TT_NUMBER);
	   color.b = tokenizer.nval;
	 } else if(tokenizer.sval.equals("filter")){
	   expect(tokenizer,StreamTokenizer.TT_NUMBER);
	   color.f = tokenizer.nval;
	 } else if((clr = findColor(tokenizer.sval))!=null){
	   color = (POVColor)clr.clone();
	   //System.out.println("color: "+ color + "clr: " + clr);
	 } else { // something different
	   tokenizer.pushBack();
	   return color;
	 }
       } else {
	 tokenizer.pushBack();
	 //exception(tokenizer,"color expected");
	 return color;
       }
     }
   }
   
   public static ColorMapEntry  
    getColorMapEntry(POVStreamTokenizer tokenizer) throws IOException {
      expect(tokenizer,'['," \"[\" expected");
      double t1=0,t2=1;
      expect(tokenizer,StreamTokenizer.TT_NUMBER);
      t1 = tokenizer.nval;
      //expect(tokenizer,',');
      tokenizer.nextToken();
      if(tokenizer.ttype != ',') // there may be ',' or may be not
	tokenizer.pushBack(); 	

      if(tokenizer.nextToken() == StreamTokenizer.TT_NUMBER){
	t2 = tokenizer.nval;         // double color entry
	POVColor c1 = getColor(tokenizer);
	POVColor c2 = getColor(tokenizer);
	expect(tokenizer,']');
	return new ColorMapEntry(c1,c2,t1,t2);
      } else {                       // single color entry
	tokenizer.pushBack();
	POVColor c1 = getColor(tokenizer);
	expect(tokenizer,']');
	//
	// we return bad entry to let caller know, that it was single 
	//  color entry
	//
	return new ColorMapEntry(c1,c1,t1,-1); 
      }


    }
  
  static POVColorMap  
    getColorMap(POVStreamTokenizer tokenizer) throws IOException {
      expect(tokenizer,StreamTokenizer.TT_WORD);
      if(!(tokenizer.sval.equals("color_map") || 
	 tokenizer.sval.equals("colour_map")))
	exception(tokenizer,"\"color_map\" expected");
      expect(tokenizer,'{');
      POVColorMap cm =  new POVColorMap();            
      ColorMapEntry cme1 = null;
      while(true){
	switch(tokenizer.nextToken()){
	default:
	  exception(tokenizer,"color_map entry expected");
	case '}':
	  return cm;
	case '[':
	  tokenizer.pushBack();
	  ColorMapEntry cme2 = getColorMapEntry(tokenizer);
	  if(cme2.t2>=0.0){  // two colors entry
	    cm.addEntry(cme2);
	  } else {  // one color entry
	    if(cme1 != null && cme1.t1 != cme2.t1){ 
	      // there is old enry new entry will have nonzero length
	      cm.addEntry(new ColorMapEntry(cme1.c1,cme2.c1,cme1.t1,cme2.t1,cme1.a1,cme2.a1));
	    } 
	    cme1 = cme2;
	  }
	  break;
	case StreamTokenizer.TT_WORD:
	  POVColorMap map;
	  if((map = findColorMap(tokenizer.sval)) != null){
	    expect(tokenizer,'}');
	    return (POVColorMap)(map.clone());
	  }
	}
      }
    }
  
  static POVColorMap  getColorMap(String str) throws IOException{
    POVColorMap cm = null;
    POVStreamTokenizer tokenizer = new 
      POVStreamTokenizer(new StringBufferInputStream(str));
    return getColorMap(tokenizer);
  }
   
   static POVPigment getPigment(POVStreamTokenizer tokenizer) throws IOException {
     expect(tokenizer,"pigment");
     expect(tokenizer,'{');
     POVPigment pigment = new POVPigment();
     while(true){
       if(tokenizer.nextToken() == '}')
	 return pigment;
       //"mandel"	
       if(tokenizer.ttype == StreamTokenizer.TT_WORD){
	 POVPigment pigm;
	 if((pigm = findPigment(tokenizer.sval)) != null){
	   pigment = (POVPigment)pigm.clone();
	 } else if(tokenizer.sval.equals("granite") || 
	    tokenizer.sval.equals("marble") || 
	    tokenizer.sval.equals("onion") || 
	    tokenizer.sval.equals("bozo") || 
	    tokenizer.sval.equals("spotted") || 
	    tokenizer.sval.equals("wood") || 
	    tokenizer.sval.equals("agate") ||
	    tokenizer.sval.equals("leopard") || 
	    tokenizer.sval.equals("wrinkles")|| 
	    tokenizer.sval.equals("crackle"))
	   pigment.type.current = tokenizer.sval;
	 else if(tokenizer.sval.equals("gradient")){
	   pigment.type.current = tokenizer.sval;
	   pigment.direction = getDirection(tokenizer);
	 } else if(tokenizer.sval.equals("color") ||
		   tokenizer.sval.equals("colour")){
	   tokenizer.pushBack();
	   pigment.color = getColor(tokenizer);
	 } else if(tokenizer.sval.equals("color_map") || 
		   tokenizer.sval.equals("colour_map")){
	   tokenizer.pushBack();
	   pigment.color_map = getColorMap(tokenizer);	   
	 } else if(tokenizer.sval.equals("turbulence")){
	   pigment.turbulence = getVector(tokenizer,3);
	 } else if(tokenizer.sval.equals("octaves")){
	   pigment.octaves = getNumber(tokenizer).value;	   
	 } else if(tokenizer.sval.equals("agate_turb")){
	   pigment.agate_turb = getNumber(tokenizer).value;
	 } else if(tokenizer.sval.equals("omega")){
	   pigment.omega = getNumber(tokenizer).value;	   
	 } else if(tokenizer.sval.equals("lambda")){
	   pigment.lambda = getNumber(tokenizer).value;
	 } else if(tokenizer.sval.equals("frequency")){
	   pigment.frequency = getNumber(tokenizer).value;	   
	 } else if(tokenizer.sval.equals("phase")){
	   pigment.phase = getNumber(tokenizer).value;	   
	 } else if(tokenizer.sval.equals("quick_color")){
	   pigment.quick_color = getColor(tokenizer);// **** bug
	 } else if(tokenizer.sval.equals("scale")){
	   pigment.scale = getVector(tokenizer,3);	   
	 } else if(tokenizer.sval.equals("rotate")){
	   pigment.rotate = getVector(tokenizer,3);	   
	 } else if(tokenizer.sval.equals("translate")){
	   pigment.translate = getVector(tokenizer,3);	   
	 } else {
	   exception(tokenizer," illegal keyword ");
	 }
       } else {
	 exception(tokenizer," illegal symbol ");
       }              
     }
   }

   
   static POVTexture
     getTextureLayer(POVStreamTokenizer tokenizer) throws IOException {
       expect(tokenizer,'{');
       POVTexture tex = new POVTexture();
       tex.addLayer(new POVTextureLayer());
       while(true){
	 if(tokenizer.nextToken() == '}')
	   return tex;
	 if(tokenizer.ttype == StreamTokenizer.TT_WORD){
	   POVTexture t;
	   if((t = findTexture(tokenizer.sval))!= null){
	     tex = (POVTexture)t.clone();
	   } else if(tokenizer.sval.equals("pigment")){
	     tokenizer.pushBack();
	     tex.lastLayer().pigment = getPigment(tokenizer);
	   } else if(tokenizer.sval.equals("normal")){
	     tokenizer.pushBack();
	     tex.lastLayer().normal = new POVNormal(tokenizer);
	   } else if(tokenizer.sval.equals("finish")){
	     tokenizer.pushBack();
	     tex.lastLayer().finish = new POVFinish(tokenizer);
	   } else if(tokenizer.sval.equals("scale")){
	     tex.scale(getVector(tokenizer,3));	   
	   } else if(tokenizer.sval.equals("rotate")){
	     tex.rotate(getVector(tokenizer,3));	   
	   } else if(tokenizer.sval.equals("translate")){
	     tex.translate(getVector(tokenizer,3));	   
	   } else {
	     exception(tokenizer," illegal keyword ");
	   }
	 } else {
	   exception(tokenizer," illegal symbol ");
	 }
       }
     }
   
   static POVTexture
     getTexture(POVStreamTokenizer tokenizer) throws IOException {
       expect(tokenizer,"texture");
       POVTexture tex =  new POVTexture();
       while(true){
	 expect(tokenizer,'{');
	 tokenizer.pushBack();
	 POVTexture t = getTextureLayer(tokenizer);// it may be several layers
	 for(int i=0; i < t.layers.size();i++){
	   tex.addLayer((POVTextureLayer)(t.layers.elementAt(i)));
	 }
	 if(tokenizer.nextToken() != StreamTokenizer.TT_WORD ||
	    !tokenizer.sval.equals("texture")){ // there may be another layer
	   tokenizer.pushBack();
	   return tex;
	 }
       }
     }
          
   public static POVDeclare 
     getDeclare(POVStreamTokenizer tokenizer) throws IOException{
       expect(tokenizer,"#declare");
       expect(tokenizer,StreamTokenizer.TT_WORD);
       String name = new String(tokenizer.sval);
       expect(tokenizer,'=');
       if(tokenizer.nextToken() == StreamTokenizer.TT_NUMBER){
	 return new POVDeclare(name,new POVNumber(tokenizer.nval));
       } else if(tokenizer.ttype == StreamTokenizer.TT_WORD){
	 if(tokenizer.sval.equals("color_map")){
	   tokenizer.pushBack();
	   return new POVDeclare(name,getColorMap(tokenizer));
	 } else if(tokenizer.sval.equals("color") ||
		   tokenizer.sval.equals("colour")){
	   tokenizer.pushBack();
	   return new POVDeclare(name,getColor(tokenizer));       
	 } else if(tokenizer.sval.equals("color_map") ||
		   tokenizer.sval.equals("colour_map")){
	   tokenizer.pushBack();
	   return new POVDeclare(name,getColorMap(tokenizer));       
	 } else if(tokenizer.sval.equals("texture")){
	   tokenizer.pushBack();
	   POVTexture tex = getTexture(tokenizer);
	   tex.setFileName(name,currentFile.getParent());
	   return new POVDeclare(name,tex);       
	 } else if(tokenizer.sval.equals("pigment")){
	   tokenizer.pushBack();
	   POVPigment pigment = getPigment(tokenizer);
	   pigment.setFileName(name,currentFile.getParent());
	   return new POVDeclare(name,pigment);       
	 } else if(tokenizer.sval.equals("finish")){
	   tokenizer.pushBack();
	   return new POVDeclare(name,new POVFinish(tokenizer));       
	 } else if(tokenizer.sval.equals("version")){
	   return new POVDeclare(name,new POVNumber(3.0));
	 } else {
	   exception(tokenizer, " unexpected keyword ");
	 }     
       } else {
	 exception(tokenizer, " unexpected symbol ");	 
       }
       return null;
     }
 


   public static POVNumber findNumber(String name){
     POVDeclare o = (POVDeclare)hashtable.get(name);
     if(o!=null && o.o instanceof POVNumber)
       return ((POVNumber)(o.o));
     else
       return null;
   }

   public static POVColor findColor(String name){
     POVDeclare o = (POVDeclare)hashtable.get(name);
     if(o!=null && o.o instanceof POVColor){
       //System.out.println(o);
       return ((POVColor)(o.o));
     }
     else
       return null;
   }

   public static POVColorMap findColorMap(String name){
     POVDeclare o = (POVDeclare)hashtable.get(name);
     if(o!=null && o.o instanceof POVColorMap)
       return (POVColorMap)(o.o);
     else 
       return null;     
   }

   public static POVTexture findTexture(String name){
     POVDeclare o = (POVDeclare)hashtable.get(name);
     if(o!=null && o.o instanceof POVTexture)
       return (POVTexture)(o.o);
     else 
       return null;          
   }

   public static POVPigment findPigment(String name){
     POVDeclare o = (POVDeclare)hashtable.get(name);
     if(o!=null && o.o instanceof POVPigment)
       return (POVPigment)(o.o);
     else 
       return null;               
   }

   public static POVFinish findFinish(String name){
     POVDeclare o = (POVDeclare)hashtable.get(name);
     if(o!=null && o.o instanceof POVFinish)
       return (POVFinish)(o.o);
     else 
       return null;               
   }

   public static POVNormal findNormal(String name){
     POVDeclare o = (POVDeclare)hashtable.get(name);
     if(o!=null && o.o instanceof POVNormal)
       return (POVNormal)(o.o);
     else 
       return null;               
   }

   static File currentFile;
   static Vector currentArray;
   static Hashtable hashtable = new Hashtable();
   static Vector array = new Vector();
   static Vector fileList = new Vector();

   public static POVDeclare addObject(POVDeclare object){
     hashtable.put(object.name,object);	   
     currentArray.addElement(object);  
     return object;
   }
   
   public static void addFile(File f){
     fileList.addElement(f);  // we will store all names of all files
     currentArray = new Vector();
     currentFile = f;
     array.addElement(currentArray);
   }

   public static Vector getCurrentArray(){
     if(currentArray == null){
       if(array.size()>0){
	 currentArray = (Vector)array.elementAt(0);
	 currentFile = (File)fileList.elementAt(0);
       }              
     }
     return currentArray;
   }
   
   public static void setCurrentFile(int n){
     if(n >= 0 && n < fileList.size()){
       currentFile = (File)fileList.elementAt(n);
       currentArray = (Vector)array.elementAt(n);
     }
   }

   public static Vector getElementsByType(String type){
     Vector elem = new Vector();
     int n = array.size();
     for(int i=0; i < n; i++){
       if(array.elementAt(i) instanceof POVDeclare){
	 POVDeclare o = (POVDeclare)array.elementAt(i);
	 //
	 // enwsWith used because of long possible package name
	 //
	 if(o.o.getClass().getName().endsWith(type)){
	   elem.addElement(o);
	 } 
       }	  
     }
     return elem;
   }
   

   /**
     creates backup files of all files, which was read
    */
   
   public static void makeBackup(File f) throws IOException {
       File fn = new File(f.getPath()+"~");
       if(fn.exists()){
	 if(fn.canWrite()){
	   fn.delete();	   
	 } else {
	   throw new IOException("unable to remove \""+fn.getPath()+"\"");
	 }
       }
       f.renameTo(fn);
     }

   public static void makeBackup() throws IOException {
     for(int i=0;i<fileList.size();i++){
       makeBackup((File)fileList.elementAt(i));
     }
   }
   
   public static void save(File f, Vector v)throws IOException {
     BufferedOutputStream fout = 
       new BufferedOutputStream(new FileOutputStream(f));
     PrintStream out = new PrintStream(fout);
     for(int j=0; j < v.size(); j++){
       POVObject o = (POVObject)v.elementAt(j);
       out.println(o);	 
       if(o instanceof POVDeclare){
	 System.out.print("writing "+((POVDeclare)o).name+"..."); System.out.flush();
	 if(((POVDeclare)o).o instanceof POVTexture){
	   POVTexture tex = (POVTexture)((POVDeclare)o).o;
	   if(tex.isModified){
	     tex.saveImage();
	   }
	 } else if(((POVDeclare)o).o instanceof POVPigment){
	   POVPigment pigm = (POVPigment)((POVDeclare)o).o;
	   if(pigm.isModified){
	     pigm.saveImage();
	   }
	 }
	 System.out.println("done");
       }     
     } 
     // do we need it?
     out.close();
     fout.close();
   }

   public static void saveCurrent()throws IOException {
     if(currentFile == null || currentArray == null){ // hmm...
       return;
     }     
     makeBackup(currentFile);
     save(currentFile,currentArray);
   }

   public static void saveAll()throws IOException {
     makeBackup();
     Vector streams = new Vector();
     for(int i=0;i<fileList.size();i++){
       File f = (File)fileList.elementAt(i);
       makeBackup(f);
       save(f,(Vector)array.elementAt(i));
     }
   }

   public static void closeCurrent(){
     if(currentFile != null && currentArray != null){
       int current = fileList.indexOf(currentFile);
       fileList.removeElementAt(current);
       
       for(int k=0; k < currentArray.size();k++ ){
	 Object o = currentArray.elementAt(k);
	 if(o instanceof POVDeclare)
	   hashtable.remove(((POVDeclare)o).name);
       }
       
       array.removeElementAt(current);
       current--;
       if(current < 0)
	 current  = 0;
       if(fileList.size() == 0){
	 currentFile = null;
	 currentArray = null;
       } else {
	 currentFile = (File)fileList.elementAt(current);
	 currentArray = (Vector)array.elementAt(current);
       }
     }
   }

   public static boolean rename(POVDeclare o, String newName){
     if(o == null)
       return false;
     if(hashtable.get(newName) != null){
       System.err.println("New name \""+newName+"\" is in use");
       return false;
     }
     hashtable.remove(o.name);
     o.rename(newName);
     hashtable.put(o.name,o);
     return true;
   }

   public static void remove(POVDeclare o){
     hashtable.remove(o.name);
     if(currentArray != null){
       int ind = currentArray.indexOf(o);
       if(ind > 0)
	 currentArray.removeElementAt(ind);
     }
   }

   /**
     parses input file as pov file and puts all objects from there 
     to Vector of Vector array (each vector for each file)
   */
   public static void parse(File f)throws IOException {     
     for(int i=0;i<fileList.size();i++){
       if(f.getPath().equals(((File)fileList.elementAt(i)).getPath())) 
	 return; // we have already read this file
     }
     File oldFile = currentFile;
     Vector oldArray = currentArray;
     addFile(f);
     currentFile = f;
     POVStreamTokenizer tokenizer = 
       new POVStreamTokenizer(new BufferedInputStream(new FileInputStream(f)));
     main_loop:while(true){
       switch(tokenizer.nextToken()){
       case StreamTokenizer.TT_EOF:
	 break main_loop;
       default:
	  exception(tokenizer,"something strange");
       case StreamTokenizer.TT_WORD:
	 if(tokenizer.sval.equals("#declare")){
	   tokenizer.pushBack();
	   POVDeclare o = getDeclare(tokenizer);
	   o.where = currentFile;
	   if(hashtable.get(o.name) != null){
	     System.err.println(o.where.getName()+":"+tokenizer.lineno()+
				" dublicate name - "+ o.name + " - ignored");
	   } else {
	     hashtable.put(o.name,o);	   
	     currentArray.addElement(o);
	   }
	   //System.out.println(o);
	 } else if(tokenizer.sval.equals("#version")){
	   getNumber(tokenizer);
	 } else if(tokenizer.sval.equals("#include")){
	   if(tokenizer.nextToken() == '\"'){
	     // find where is this file really
	     // we will assume now, that everything should be in 
	     // the same directory
	     POVInclude inc = new POVInclude(new File(f.getParent(),
						      tokenizer.sval),
					     currentFile);
	     currentArray.addElement(inc);
	     parse(inc.f);
	   } else 
	     exception(tokenizer," file name expected");
	 }
       }
     }
     currentFile = oldFile;
     currentArray = oldArray;
   }

   public static void main(String[] args)throws IOException {
     if(args.length >0){
       File f = new File(args[0]);
       parse(f);
       for(int i=0;i<array.size();i++){
	 POVDeclare o = (POVDeclare)array.elementAt(i);
	 if(o.o instanceof POVTexture)
	 System.out.println(o);
       }
       /*
       Enumeration enum = hashtable.elements();
       while(enum.hasMoreElements()){
	 POVDeclare po = (POVDeclare)(enum.nextElement());
	 System.out.println("#declare "+po.name + "=" + po.o);       
       }
       */
     }
   }
 }
