package PVS.POVTextureEditor;
// POVNormal.java - a POVray  Texture editor
//
// Copyright (C) 1996 by Vladimir Bulatov <V.Bulatov@ic.ac.uk>.  
//            All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
import java.io.*;

import PVS.Utils.*;

public class POVNormal extends Object{

  static String[] normalNames = {
    "none","bumps", "dents", "ripples", "waves", "wrinkles","granite"};


  public ChoiceR type = new ChoiceR(normalNames,normalNames[0]);

  public DoubleR   octaves = new DoubleR(6.);
  public DoubleR   omega = new DoubleR(0.5);
  public DoubleR   lambda = new DoubleR(2.);
  public DoubleR   strength = new DoubleR(0.0);
  public DoubleR   frequency = new DoubleR(1.);
  public DoubleR   phase = new DoubleR(0.);
  public DoubleR[] turbulence = new DoubleR[3];
  public DoubleR[] scale = new DoubleR[3];
  public DoubleR[] rotate = new DoubleR[3];
  public DoubleR[] translate = new DoubleR[3];;

  POVNormal(){
    initVectors();
  }
  
  public void initVectors(){
    for(int i=0;i<3;i++){
      scale[i] = new DoubleR(1.);
      rotate[i] = new DoubleR(0.);
      turbulence[i] = new DoubleR(0.);
      translate[i] = new DoubleR(0.);
    }
  }

  POVNormal(POVStreamTokenizer tokenizer) throws IOException {
    initVectors();
    POVParse.expect(tokenizer,"normal");
    POVParse.expect(tokenizer,'{');
    int brack = 1;
    while(true){ // skeep everything
      if(tokenizer.nextToken() == '}')
	return;
       if(tokenizer.ttype == StreamTokenizer.TT_WORD){
	 POVNormal norm;
	 if((norm = POVParse.findNormal(tokenizer.sval)) != null){
	   this.assign(norm);
	 } else if(normalNames[1].equals(tokenizer.sval) ||
		   normalNames[2].equals(tokenizer.sval) ||
		   normalNames[3].equals(tokenizer.sval) ||
		   normalNames[4].equals(tokenizer.sval) ||
		   normalNames[5].equals(tokenizer.sval) ||
		   normalNames[6].equals(tokenizer.sval)){
	   type.current = tokenizer.sval;
	   strength = POVParse.getNumber(tokenizer).value;
	 } else if(tokenizer.sval.equals("turbulence")){
	   turbulence = POVParse.getVector(tokenizer,3);
	 } else if(tokenizer.sval.equals("octaves")){
	   octaves = POVParse.getNumber(tokenizer).value;	   
	 } else if(tokenizer.sval.equals("omega")){
	   omega = POVParse.getNumber(tokenizer).value;	   
	 } else if(tokenizer.sval.equals("lambda")){
	   lambda = POVParse.getNumber(tokenizer).value;
	 } else if(tokenizer.sval.equals("frequency")){
	   frequency = POVParse.getNumber(tokenizer).value;	   
	 } else if(tokenizer.sval.equals("phase")){
	   phase = POVParse.getNumber(tokenizer).value;	   
	 } else if(tokenizer.sval.equals("scale")){
	   scale = POVParse.getVector(tokenizer,3);	   
	 } else if(tokenizer.sval.equals("rotate")){
	   rotate = POVParse.getVector(tokenizer,3);	   
	 } else if(tokenizer.sval.equals("translate")){
	   translate = POVParse.getVector(tokenizer,3);	   
	 } else {
	   POVParse.exception(tokenizer," illegal keyword ");
	 }
       } else {
	 POVParse.exception(tokenizer," illegal symbol ");
       }                    
    }
  }

  public void assign(POVNormal norm){
    type.current = norm.type.current;
    strength.value = norm.strength.value;
    octaves.value = norm.octaves.value;
    omega.value = norm.omega.value;
    lambda.value = norm.lambda.value;
    frequency.value = norm.frequency.value;
    phase.value = norm.phase.value;

    for(int i=0;i<3;i++){
      turbulence[i].value = norm.turbulence[i].value;
      scale[i].value = norm.scale[i].value;
      rotate[i].value = norm.rotate[i].value;
      translate[i].value = norm.translate[i].value;
    }     
  }
  
  public Object clone(){
    POVNormal p = new POVNormal();
    p.type.current = type.current;
    p.strength.value = strength.value;
    p.octaves.value = octaves.value;
    p.omega.value = omega.value;
    p.lambda.value = lambda.value;
    p.frequency.value = frequency.value;
    p.phase.value = phase.value;

    for(int i=0;i<3;i++){
      p.turbulence[i].value = turbulence[i].value;
      p.scale[i].value = scale[i].value;
      p.rotate[i].value = rotate[i].value;
      p.translate[i].value = translate[i].value;
    }
    return p;
  }

  public String toString(){
    if(type.current.equals("none"))
      return "";
    StringBuffer buf = new StringBuffer("normal {");
    buf.append(type.current+" "+strength.value);
    if(turbulence[0].value != 0. || turbulence[1].value != 0. || 
       turbulence[2].value != 0.){
      buf.append(" turbulence <" + turbulence[0].value + "," 
		 + turbulence[1].value+","+turbulence[2].value+">");	  
      buf.append(" octaves " + octaves.value + " omega " + omega.value);      
      buf.append(" lambda " + lambda.value);
    }
    if(frequency.value != 1.0)
      buf.append(" frequency " + frequency.value);
    if(phase.value != 0.0)
      buf.append(" phase " + phase.value);
    if(scale[0].value != 1.0 || scale[1].value != 1.0 || scale[2].value != 1.0)
      buf.append("\nscale <"+scale[0].value+","+scale[1].value+","+ 
		 scale[2].value+">");
    if(rotate[0].value != 0.0 || rotate[1].value != 0.0 || rotate[2].value != 0.0)
      buf.append("\nrotate <"+rotate[0].value+","+rotate[1].value+","+ 
		 rotate[2].value+">");
    if(translate[0].value != 0.0 || translate[1].value != 0.0 || 
       translate[2].value != 0.0)
      buf.append("\ntranslate <"+translate[0].value+","+
		 translate[1].value+","+ +translate[2].value+">");
    buf.append("\n}");
    return new String(buf);
  }
}

