package PVS.POVTextureEditor;
// Main.java - (main program of POVray Texture editor)
//
// Copyright (C) 1996 by Vladimir Bulatov <V.Bulatov@ic.ac.uk>.  
//            All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.


import java.awt.*;
import java.io.*;
import java.util.Vector;
import java.util.Properties;
import java.util.StringTokenizer;

import PVS.Utils.*;

public class Main extends Frame{

  static final int MAX_NAME_LENGTH = 20;
  
  GridBagLayout layout = new GridBagLayout();

  List objectList = new List(10,false);

  SamplePanel samplePanel = new SamplePanel();

  POVDeclare currentObject = null;

  ColorMapEditor colorMapEditor = null;
  TextureEditor textureEditor = null;
  ColorEditor colorEditor = null;
  PigmentEditorFrame pigmentEditor = null;

  DrawableBrowser browser = null;
  
  Frame colorMapFrame = null;
  Frame textureFrame = null;
  Frame colorFrame = null;
  Frame pigmentFrame = null;
  Frame browserFrame = null;

  MenuItem[] fileMenuItem = new MenuItem[4];
  Menu menuWindow;

  MenuItem  menuEditEdit = new MenuItem("Edit...");
  MenuItem  menuEditCopy = new MenuItem("Copy");
  MenuItem  menuEditRemove = new MenuItem("Remove");
  MenuItem  menuEditRename = new MenuItem("Rename...");
  MenuItem  menuEditNew = new MenuItem("New..."); 
  MenuItem  menuEditUpdateAll = new MenuItem("Update All");

  MenuItem  menuFileLoad = new MenuItem("Load...");
  MenuItem  menuFileClose = new MenuItem("Close");
  MenuItem  menuFileSave   = new MenuItem("Save");
  MenuItem  menuFileSaveAll  = new MenuItem("Save All");
  MenuItem  menuFileBrowse   = new MenuItem("Browse");
  MenuItem  menuFileProperties  = new MenuItem("Properties...");
  MenuItem  menuFileQuit    = new MenuItem("Quit");


  FileDialog file_load_dialog = null,file_save_dialog = null;
  NewPOVObjectDialog newDialog = null;
  RenameDialog renameDialog = null;
  
  public Main(String name){
    super(name);
    init();
  }
  
  void getProperties(){
    Properties props = new Properties(System.getProperties());
    props.put("povray.path", "povray");
    props.put("gamma", "true");
    props.put("gamma.value", ""+(ImageTGA.red_gamma));
    props.put("home",Defaults.getPropertiesDir());
    props.put("file.0", "");
    props.put("file.1", "");
    props.put("file.2", "");
    props.put("file.3", "");
    // try to load local properties file
    try {
      FileInputStream in = 
	new FileInputStream(Defaults.getPropertiesDir()+
			    File.separator+"properties");
      props.load(in);
      in.close();
    } catch (Exception e) {
      System.err.println("[no properties loaded, using defaults]");
      System.err.println("[check path to povray in File|properties]");
      System.setProperties(props);
      saveProperties();
      Defaults.createDefaultFiles();
    }    
    // Install a property list.
    System.setProperties(props);

    //
    // do some one shot initialization 
    //
    String gamma = props.getProperty("gamma");
    if(gamma.equals("true"))
      ImageTGA.doGamma = true;
    else 
      ImageTGA.doGamma = false; 
    ImageTGA.red_gamma = ImageTGA.green_gamma = ImageTGA.blue_gamma = 
      (Double.valueOf(props.getProperty("gamma.value"))).doubleValue();
  }

  void	saveProperties(){
    
    Properties prop = System.getProperties();

    addPropertyRectangle(prop,"MainFrame",this);
    addPropertyRectangle(prop,"TextureFrame",textureFrame);
    addPropertyRectangle(prop,"BrowserFrame",browserFrame);
    addPropertyRectangle(prop,"ColorMapFrame",colorMapFrame);
    addPropertyRectangle(prop,"ColorFrame",colorFrame);    
    
    try {
      File f = new File(Defaults.getPropertiesDir());
      if(!f.exists())
	f.mkdirs();
      FileOutputStream out = 
	new FileOutputStream(Defaults.getPropertiesDir()+
			     File.separator+"properties");
      prop.save(out, "POV-ray Texture Editor; author V.Bulatov@ic.ac.uk\n#"); 
      out.close();      

    } catch (Exception e) {
      e.printStackTrace(System.err);
      System.err.println("[cat't save properties]");
    }    
  }

  void addPropertyRectangle(Properties props, String name, Component comp){
    Rectangle r;
    if(comp != null){      
      r = comp.bounds();
      props.put(name, String.valueOf(r.x)+" "+ String.valueOf(r.y)+" "+
		String.valueOf(r.width)+" " + String.valueOf(r.height));
    }
  }

  Rectangle getPropertyRectangle(Properties props, String name){
    String str = props.getProperty(name);
    if(str == null)
      return null;
    StringTokenizer tok = new StringTokenizer(str," ");
    Rectangle r = new Rectangle();
    if(tok.hasMoreTokens()){
      r.x = Integer.valueOf(tok.nextToken()).intValue();
      if(tok.hasMoreTokens()){
	r.y = Integer.valueOf(tok.nextToken()).intValue();
	if(tok.hasMoreTokens()){
	  r.width = Integer.valueOf(tok.nextToken()).intValue();
	  if(tok.hasMoreTokens()){
	    r.height = Integer.valueOf(tok.nextToken()).intValue();
	  }
	}
      }
    }
    return r;
  }

  void moveToPlace(Window window, String name){
    Rectangle r;
    if((r = getPropertyRectangle(System.getProperties(),name)) != null){
      window.reshape(r.x,r.y,r.width,r.height);
    } else {
      window.pack();    
    }
  }

  public void init() {

    getProperties();

    MenuBar menubar = new MenuBar();
    Menu file = new Menu("File", true);

    file.add(menuFileLoad); 
    file.add(menuFileClose);menuFileClose.disable();    
    file.add(menuFileSave);menuFileSave.disable();
    file.add(menuFileSaveAll);menuFileSaveAll.disable();
    file.add(menuFileBrowse);menuFileBrowse.disable();
    file.add(menuFileProperties);
    file.add(menuFileQuit);
    file.addSeparator();

    for(int i=0; i<4; i++){
      String name = System.getProperties().getProperty("file."+i);
      if(name != null){
	if(name.length() > MAX_NAME_LENGTH)
	  name = "..."+name.substring(name.length()-MAX_NAME_LENGTH);
	fileMenuItem[i] = new MenuItem(""+i+")"+name);
	file.add(fileMenuItem[i]);
      }
    }

    menubar.add(file);

    Menu edit = new Menu("Edit", true);
    edit.add(menuEditEdit);menuEditEdit.disable();
    edit.add(menuEditCopy);menuEditCopy.disable();
    edit.add(menuEditRemove);menuEditRemove.disable();
    edit.add(menuEditRename);menuEditRename.disable();
    edit.add(menuEditNew);  
    edit.addSeparator();
    edit.add(menuEditUpdateAll);menuEditUpdateAll.disable();
    menubar.add(edit);

    menuWindow = new Menu("Window", true); menuWindow.disable();
    
    
    menubar.add(menuWindow);

    WindowUtils.getMainWindow(this).setMenuBar(menubar);
      
    this.setLayout(layout);

    WindowUtils.constrain(this,getObjectPanel(),0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);    
  }

  void doColorEditor(){
    WindowUtils.getMainWindow(this).setCursor(Frame.WAIT_CURSOR);
    Toolkit.getDefaultToolkit().sync();
    colorEditor = new ColorEditor(new ColorExt());
    colorFrame = new CloseAbleFrame("Color");
    colorFrame.add("Center",colorEditor);
    moveToPlace(colorFrame,"ColorFrame");
    WindowUtils.getMainWindow(this).setCursor(Frame.DEFAULT_CURSOR);
  }

  void doColorMapEditor(){
    WindowUtils.getMainWindow(this).setCursor(Frame.WAIT_CURSOR);
    Toolkit.getDefaultToolkit().sync();
    colorMapEditor  = new ColorMapEditor();
    colorMapEditor.init(); // it is Applet
    colorMapFrame = new CloseAbleFrame("ColorMap");
    colorMapFrame.add("Center", colorMapEditor);
    moveToPlace(colorMapFrame,"ColorMapFrame");
    WindowUtils.getMainWindow(this).setCursor(Frame.DEFAULT_CURSOR);
  }

  void doTextureEditor(){
    WindowUtils.getMainWindow(this).setCursor(Frame.WAIT_CURSOR);
    Toolkit.getDefaultToolkit().sync();
    textureEditor = new TextureEditor(new POVTexture());
    textureFrame = new CloseAbleFrame("Texture");
    textureFrame.add("Center", textureEditor);
    moveToPlace(textureFrame,"TextureFrame");
    WindowUtils.getMainWindow(this).setCursor(Frame.DEFAULT_CURSOR);
  }
  
  void doPigmentEditor(){
    WindowUtils.getMainWindow(this).setCursor(Frame.WAIT_CURSOR);
    Toolkit.getDefaultToolkit().sync();
    pigmentEditor = new PigmentEditorFrame(new POVPigment());
    pigmentFrame = new CloseAbleFrame("Pigment");
    pigmentFrame.add("Center",pigmentEditor);
    moveToPlace(pigmentFrame,"PigmentFrame");
    WindowUtils.getMainWindow(this).setCursor(Frame.DEFAULT_CURSOR);
  }

  void doBrowser(){
    WindowUtils.getMainWindow(this).setCursor(Frame.WAIT_CURSOR);
    Toolkit.getDefaultToolkit().sync();
    browser = new DrawableBrowser();
    browserFrame = new CloseAbleFrame("Browser");
    browserFrame.add("Center",browser);
    moveToPlace(browserFrame,"BrowserFrame");
    WindowUtils.getMainWindow(this).setCursor(Frame.DEFAULT_CURSOR);
  }

  public Component createButtonsPanel(){

    Panel pButtons = new BorderPanel();pButtons.setLayout(layout);
    String[] title = {"POVRay TextureEditor",
		      "Copyright (C) 1996 by V.Bulatov",
		      "V.Bulatov@ic.ac.uk"};
    Font font = new Font("Dialog",Font.PLAIN,10);
    for(int i=0;i<3;i++){
      Label lab = new Label(title[i]);
      lab.setFont(font);
      WindowUtils.constrain(pButtons,lab,0,i,1,1,
			  GridBagConstraints.NONE,
                          GridBagConstraints.CENTER,1.,1.);
    }
    return pButtons;
  }

  public Panel getObjectPanel(){

    Panel panel = new Panel(); panel.setLayout(layout);
    
    Panel pSample = new BorderPanel();pSample.setLayout(layout);
    WindowUtils.constrain(pSample,samplePanel,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);
    
    WindowUtils.constrain(panel,objectList,0,0,1,2,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,0.2,1.);
    WindowUtils.constrain(panel,pSample,1,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);
    WindowUtils.constrain(panel,createButtonsPanel(),1,1,1,1,
			  GridBagConstraints.VERTICAL,
                          GridBagConstraints.CENTER,1.,0.2);
    return panel;
  }  

  
  public void start(){

  }

  public void stop(){

  }

  public boolean handleEvent(Event e){   

    if(e.target == objectList){
      switch(e.id){
      case Event.LIST_SELECT:
	String item = objectList.getItem(((Integer)e.arg).intValue());
	currentObject = (POVDeclare)POVParse.hashtable.get(item);
	samplePanel.setDrawable(currentObject);
	
	menuEditRename.enable();
	menuEditEdit.enable();
	menuEditNew.enable();
	menuEditCopy.enable();
	menuEditRemove.enable();

	break;
      case Event.LIST_DESELECT:
	menuEditRename.disable();
	menuEditEdit.disable();
	menuEditNew.disable();
	menuEditCopy.disable();
	menuEditRemove.disable();

	break;
      }
    }
    return super.handleEvent(e); // to pass event to action()
  }

  public boolean action(Event e, Object what){   

    for(int i=0;i<4;i++){
      if(e.target == fileMenuItem[i]){
	doLoad(System.getProperties().getProperty("file."+i));
	return true;
      }
    }

    for(int i=0; i < menuWindow.countItems();i++){
      if(e.target == menuWindow.getItem(i)){
	POVParse.setCurrentFile(i);
	initLists();
	return true;
      }
    }

    if(e.target == objectList){
      currentObject = (POVDeclare)POVParse.hashtable.get((String)what);
      samplePanel.setDrawable(currentObject);
      doEdit(currentObject);
    } else if(e.target == newDialog){ // new Dialog was closed with OK
      doCreateNew();
    } else if(e.target == renameDialog){ // new Dialog was closed with OK
      doRename();
    } else if(e.target == browser){ // user double clicked in browser
      doEdit(what);
    } else if(e.target == menuEditNew){
      newDialog = new NewPOVObjectDialog(WindowUtils.getMainWindow(this),
					 "Create New Object",true ,this);
      newDialog.pack();
      newDialog.show(); // it doesn't blocks here :-(            
    } else if(e.target == menuEditRemove){
      doRemove();
    } else if(e.target == menuEditRename){
      if(objectList.getSelectedIndex() >=0){
	doRenameDialog();
      }
    } else if(e.target == menuFileBrowse){
      if(browser == null)
	doBrowser();
      initBrowser();
    } else if(e.target == menuEditEdit){
      doEdit();
    } else if(e.target == menuEditUpdateAll){
      doUpdateAll();
    } else if(e.target == menuFileLoad){
      doLoad();
      return true;
    } else if(e.target == menuFileProperties){
      TextureEditorProperties prop = new 
	TextureEditorProperties(this,"TextureEditor properties",true);
      prop.pack();
      prop.show();
      return true;	
    } else if(e.target == menuEditCopy){
      doCopy(); return true;
    } else if(e.target == menuFileClose){
      doClose(); return true;
    } else if(e.target == menuFileSave){
      doSave(); return true;
    } else if(e.target == menuFileSaveAll){
      doSaveAll(); return true;
    } else if(e.target == menuFileQuit){
      saveProperties();
      System.exit(0);
    }
    return false;
  }

  void disableFileMenu(){
    menuFileClose.disable();    
    menuFileSave.disable();
    menuFileSaveAll.disable();
    menuFileBrowse.disable();
    menuEditUpdateAll.disable();
  }

  void enableFileMenu(){
    menuFileClose.enable();
    menuFileSave.enable();
    menuFileSaveAll.enable();
    menuFileBrowse.enable(); 
    menuEditUpdateAll.enable();
  }
  
  void doClose(){
    POVParse.closeCurrent();
    initLists();
    initMenuWindow();
    if(POVParse.getCurrentArray() == null){
      menuWindow.disable();
      disableFileMenu();
    }
    System.gc(); // garbage collection 
  }

  void doRename(){ // called, when rename dialog was closed with OK
    if(POVParse.rename(currentObject,renameDialog.newName.value)){
      int ind = objectList.getSelectedIndex();
      objectList.replaceItem(currentObject.name,ind);
      objectList.select(ind);
      samplePanel.repaint();
    }
  }

  void doRenameDialog(){
    if(currentObject == null)
      return;
    renameDialog = new RenameDialog(WindowUtils.getMainWindow(this),
					   "Rename Object",true ,this,currentObject);
    renameDialog.pack();
    renameDialog.show();
  }

  void doRemove(){
    int ind = objectList.getSelectedIndex();
    if(ind >= 0){
      objectList.delItem(ind);
      POVParse.remove(currentObject);
    }
  }

  /**
    
   */
  void doUpdateAll(){
    try{
      for(int i=0; i < POVParse.array.size(); i++){
	Vector v = (Vector)POVParse.array.elementAt(i);
	for(int j=0; j < v.size(); j++){
	  POVObject ob = (POVObject)v.elementAt(j);
	  if(ob instanceof POVDeclare){
	    POVDeclare o = (POVDeclare)ob;
	    if(o.o instanceof POVTexture){ 
	      ((POVTexture)(o.o)).makeThumbnail(null);
	    } else if(o.o instanceof POVPigment){ 
	      ((POVPigment)(o.o)).makeThumbnail(null);
	    }
	  }
	}  
	
      }
    } catch(Exception e){
      e.printStackTrace();
    }
  }
  /**
    
   */
  void initBrowser() {
    Vector array = new Vector();
    Vector v = POVParse.getCurrentArray();
    for(int i = 0; i < v.size();i++){
      Object o = v.elementAt(i);
      if(o instanceof Drawable)
	array.addElement(o);
    }
    browser.init(array,this); browserFrame.pack(); browserFrame.show();
  }


  /**
    
   */
  void doEdit(){
    if(currentObject != null)
      doEdit(currentObject);
  }

  void doEdit(Object ob){

    if(!(ob instanceof POVDeclare))
      return;

    POVDeclare o = (POVDeclare)ob;
    if(o.o instanceof POVColor){
      if(colorEditor == null)
	doColorEditor();
      colorEditor.init(o);
    } else if(o.o instanceof POVTexture){
      if(textureEditor == null)
	doTextureEditor();
      textureEditor.init(o);
    } else if(o.o instanceof POVColorMap){
      if(colorMapEditor == null)
	doColorMapEditor();
      colorMapEditor.colorMapPanel.init(o);
      colorMapEditor.start(); // it has additional frame
    } else if(o.o instanceof POVPigment){
      if(pigmentEditor == null)
	doPigmentEditor();
      pigmentEditor.init(o);
    }
  }

  /**
    
   */
  void doCopy(){
    if(currentObject == null)
      return;
    if( currentObject.o instanceof POVTexture){
      Clipboard.put(((POVTexture)currentObject.o).clone());
    } else if( currentObject.o instanceof POVPigment){
      Clipboard.put(((POVPigment)currentObject.o).clone());
    } else if( currentObject.o instanceof POVColor){
      Clipboard.put(((POVColor)currentObject.o).clone());
    } else if( currentObject.o instanceof POVColorMap){
      Clipboard.put(((POVColorMap)currentObject.o).clone());
    }
  }
  
  /**
    
   */
  void doCreateNew(){
    File file = null;
    if(newDialog.fileChoice.current.equals("new file")){
      if(file_save_dialog == null){
	file_save_dialog = 
	  new FileDialog(WindowUtils.getMainWindow(this),
			 "Place "+newDialog.objectName.value+" to ...",
			 FileDialog.SAVE);
      } else {
	file_save_dialog. setTitle("Place "+newDialog.objectName.value+
				   " to ...");
      }
      file_save_dialog.pack();
      file_save_dialog.show();
      
      String fname = file_save_dialog.getFile();
      String fpath = file_save_dialog.getDirectory();
      
      if(fname != null && fpath != null){  // user had selected something
	file = new File(fpath,fname);
	if(file.exists()){
	  System.out.println("file:"+fpath+fname+" exists");
	  return;
	}
	POVParse.addFile(file);
      } else {
	return;
      }
    } else {
      // search of real file with given name
      for(int i=0;i<POVParse.fileList.size();i++){
	File f = (File)POVParse.fileList.elementAt(i);
	if(f.getName().equals(newDialog.fileChoice.current)){
	  file = f;
	  break;
	}
      }
      if(file == null) // it shouldn't happens
	return;
    }

    //
    // actual object creation
    //
    POVDeclare object = null;
    if(newDialog.newType.current.equals("texture")){
      object = new POVDeclare(newDialog.objectName.value,new POVTexture(),file);
      // texture can have thumbnail and should know it location
      ((POVTexture)object.o).setFileName(object.name,file.getParent());
    } else if(newDialog.newType.current.equals("color")){
      object = new POVDeclare(newDialog.objectName.value, new POVColor(),file);
    } else if(newDialog.newType.current.equals("color_map")){
      object =new POVDeclare(newDialog.objectName.value,new POVColorMap(),file);
    } else if(newDialog.newType.current.equals("pigment")){
      object =new POVDeclare(newDialog.objectName.value, new POVPigment(),file);
      // pigment can have thumbnail and should know it location
      ((POVPigment)object.o).setFileName(object.name,file.getParent());
    }
    if(object != null){
      POVParse.addObject(object);	
      initLists();
      initMenuWindow();
      menuWindow.enable();
      enableFileMenu();
    }
  }
    
  
  /**
    
   */
  void doSaveAll() {
    WindowUtils.getMainWindow(this).setCursor(Frame.WAIT_CURSOR);
    Toolkit.getDefaultToolkit().sync();
    try {
      POVParse.saveAll();
    } catch(IOException ex){
      ex.printStackTrace(System.out);	  
    } finally{
      WindowUtils.getMainWindow(this).setCursor(Frame.DEFAULT_CURSOR);
    }
  }

  /**
    
   */
  void doSave() {
    WindowUtils.getMainWindow(this).setCursor(Frame.WAIT_CURSOR);
    Toolkit.getDefaultToolkit().sync();
    try {
      POVParse.saveCurrent();
    } catch(IOException ex){
      ex.printStackTrace(System.out);	  
    } finally{
      WindowUtils.getMainWindow(this).setCursor(Frame.DEFAULT_CURSOR);
    }
  }
  
  /**
    
   */

  static String[] povExtensions = new String[6];
  static ExtensionFileNameFilter povExtensionFilter;
  static{
    povExtensions[0] = ".pov";
    povExtensions[1] = ".inc";
    povExtensions[2] = ".INC";
    povExtensions[3] = ".POV";
    povExtensions[4] = ".map";
    povExtensions[5] = ".MAP";
    povExtensionFilter = new ExtensionFileNameFilter(povExtensions);
  }

  public void readFile(File f) throws IOException {
    POVParse.parse(f);
    POVParse.setCurrentFile(POVParse.fileList.size()-1);

    initLists();
    
    initMenuWindow();
    menuWindow.enable();
    enableFileMenu();
  }

  void initMenuWindow(){
    while(menuWindow.countItems() > 0)
      menuWindow.remove(0);
    for(int i=0; i < POVParse.fileList.size();i++){
      String name = ((File)POVParse.fileList.elementAt(i)).getPath();
      if(name.length() > MAX_NAME_LENGTH)
	name = "..."+name.substring(name.length()-MAX_NAME_LENGTH);      
      menuWindow.add(name);      
    }
  }

  public void initLists(){

    if(objectList.countItems() > 0)
      objectList.clear();
    currentObject = null;
    Vector v = POVParse.getCurrentArray();
    if(v == null){
      WindowUtils.getMainWindow(this).setTitle("Texture Editor ");
      return;
    }
    for(int i=0; i < v.size(); i++){
      Object ob = v.elementAt(i);
      if(ob instanceof POVDeclare){
	POVDeclare o = (POVDeclare)ob;
	objectList.addItem(o.name);
      }
    }
    WindowUtils.getMainWindow(this).setTitle("Texture Editor ("+
					     POVParse.currentFile.getName()+")");
    this.pack();
  }


  void doLoad(String path){
    File f = new File(path);
    WindowUtils.getMainWindow(this).setCursor(Frame.WAIT_CURSOR);
    Toolkit.getDefaultToolkit().sync();
    try {
      this.readFile(f);
    } catch(IOException ex){
      ByteArrayOutputStream out = new ByteArrayOutputStream();
      PrintStream prn = new PrintStream(out);
      ex.printStackTrace(prn);
      Frame frame = new DestroyableFrame("parse error");
      TextArea text = new TextArea( out.toString());
      text.setFont(new Font("Dialog",Font.PLAIN,14));
      frame.add("Center",text); 
      frame.pack(); 
      frame.show();
    } finally {
      WindowUtils.getMainWindow(this).setCursor(Frame.DEFAULT_CURSOR);
    }
    
  }
  
  void doLoad(){
    if(file_load_dialog == null){
      file_load_dialog = new FileDialog(WindowUtils.getMainWindow(this), 
					"Load Texture Libarary", 
					FileDialog.LOAD);
    }
    
    // doesn't works 
    file_load_dialog.setFilenameFilter(povExtensionFilter);
    file_load_dialog.pack();
    file_load_dialog.show(); 

    String fname = file_load_dialog.getFile();
    String fpath = file_load_dialog.getDirectory();
    String path = fpath+fname;
    
    if(fname != null && fpath != null){  // user had selected something

      // push down old file names
      Properties props = System.getProperties();
      for(int i=3; i>0; i--){
	String name = props.getProperty("file."+(i-1));
	props.put("file."+i,name);
      }
      props.put("file.0",path);

      for(int i=0; i < 4; i++){
	String name = props.getProperty("file."+i);
	if(name.length() > MAX_NAME_LENGTH)
	  name = "..."+name.substring(name.length()-MAX_NAME_LENGTH);
	fileMenuItem[i].setLabel(""+i+")"+name);
      }
      System.setProperties(props);
            
      doLoad(fpath+fname);
    }
  }

  public String getAppletInfo() {
    return getClass().getName() + 
      " - Copyright (C) 1996 by Vladimir Bulatov <V.Bulatov.ic.ac.uk>.\n"+
	"  All rights reserved.";
  }

  public static void main(String[] args) throws IOException{
    //ArchiveManager applet = new ArchiveManager();
    Main frame = new Main("Texture Editor");
    //frame.add("Center", applet);
    //frame.init();
    if(args.length>0){
      File f = new File(args[0]);
      System.out.println(f.getPath()+" "+ f.getName()+" "+f.getAbsolutePath()+
			 " "+f.getPath()+" "+f.getParent());
      frame.readFile(new File(args[0]));
    }
    Rectangle r;
    if((r = frame.getPropertyRectangle(System.getProperties(),"MainFrame")) != null){
      frame.reshape(r.x,r.y,r.width,r.height);
    } else {
      frame.pack();
    }
    frame.show();
    frame.start();
  }
}

class NewPOVObjectDialog extends Dialog{

  static String[] newTypes = {"texture","color", "pigment", "color_map"};

  public ChoiceR newType = new ChoiceR(newTypes,newTypes[0]);
  String[] fileList;
  ChoiceR fileChoice;
  StringR objectName = new StringR("unnamed");

  Button buttonOK = new Button("OK");
  Button buttonCancel = new Button("Cancel");
  String result = null;
  Component notifyParent = null;

  ChoiceField   typeField;
  ChoiceField   fileField;
  StringField   nameField;

  GridBagLayout layout = new GridBagLayout();

  NewPOVObjectDialog(Frame frame,String title,boolean modal,
		     Component notifyParent){
    super(frame,title,modal);

    fileList = new String[POVParse.fileList.size()+1];    
    for(int i=0;i < fileList.length-1;i++){
      fileList[i] = ((File)POVParse.fileList.elementAt(i)).getName();
    }
    fileList[fileList.length-1] = "new file";
    fileChoice = new ChoiceR(fileList,fileList[0]);

    this.notifyParent = notifyParent;
    
    this.setLayout(layout);

    typeField = new ChoiceField("create:",newType,layout);
    nameField = new StringField("name:",objectName,layout,16);
    fileField = new ChoiceField("place to:",fileChoice,layout);

    Panel panel1 = new BorderPanel();panel1.setLayout(layout);
    WindowUtils.constrain(panel1,typeField,0,0,1,1, GridBagConstraints.NONE, 
			  GridBagConstraints.CENTER,1.,1.,
			  5,5,5,5);
    WindowUtils.constrain(panel1,nameField,0,1,1,1, GridBagConstraints.NONE, 
			  GridBagConstraints.CENTER,1.,1.,
			  5,5,5,5);
    WindowUtils.constrain(panel1,fileField,0,2,1,1, GridBagConstraints.NONE, 
			  GridBagConstraints.CENTER,1.,1.,
			  5,5,5,5);
    Panel panel2 = new BorderPanel();panel2.setLayout(layout);
    WindowUtils.constrain(panel2,buttonOK,0,0,1,1, GridBagConstraints.NONE,
			  GridBagConstraints.CENTER,1.,1.);
    WindowUtils.constrain(panel2,buttonCancel,1,0,1,1, GridBagConstraints.NONE, 
			  GridBagConstraints.CENTER,1.,1.);

    WindowUtils.constrain(this,panel1,0,0,1,1, GridBagConstraints.BOTH, 
			  GridBagConstraints.CENTER,1.,1.);
    WindowUtils.constrain(this,panel2,0,1,1,1, GridBagConstraints.BOTH, 
			  GridBagConstraints.CENTER,1.,1.);
  }
  
  public boolean action(Event e, Object o){
    if(e.target == buttonOK){
      result = "OK";
      this.hide();
      Toolkit.getDefaultToolkit().sync();      
      notifyParent.postEvent(new Event(this,Event.ACTION_EVENT,null));
      //this.dispose();
      return true;
    } else if(e.target == buttonCancel){
      this.hide();
      return true;
    }
    return false;
  }  
}

class RenameDialog extends Dialog{

  StringR newName = new StringR("");
  StringR oldName = new StringR("");

  Button buttonOK = new Button("OK");
  Button buttonCancel = new Button("Cancel");
  Component notifyParent = null;

  StringField   nameField;
  StringField   oldnameField;

  GridBagLayout layout = new GridBagLayout();

  RenameDialog(Frame frame,String title,boolean modal,
		     Component notifyParent,POVDeclare object){
    super(frame,title,modal);

    this.notifyParent = notifyParent;
    
    this.setLayout(layout);

    oldName.value = object.name;
    newName.value = object.name;

    oldnameField = new StringField("rename:",oldName,layout,16);
    oldnameField.textField.setEditable(false);
    nameField = new StringField("to:",newName,layout,16);

    Panel panel1 = new BorderPanel();panel1.setLayout(layout);

    WindowUtils.constrain(panel1,oldnameField,0,0,1,1, GridBagConstraints.NONE, 
			  GridBagConstraints.EAST,1.,1.);
    WindowUtils.constrain(panel1,nameField,0,1,1,1, GridBagConstraints.NONE, 
			  GridBagConstraints.EAST,1.,1.);

    Panel panel2 = new BorderPanel();panel2.setLayout(layout);
    WindowUtils.constrain(panel2,buttonOK,0,0,1,1, GridBagConstraints.NONE,
			  GridBagConstraints.CENTER,1.,1.);
    WindowUtils.constrain(panel2,buttonCancel,1,0,1,1, GridBagConstraints.NONE, 
			  GridBagConstraints.CENTER,1.,1.);

    WindowUtils.constrain(this,panel1,0,0,1,1, GridBagConstraints.BOTH, 
			  GridBagConstraints.CENTER,1.,1.);
    WindowUtils.constrain(this,panel2,0,1,1,1, GridBagConstraints.BOTH, 
			  GridBagConstraints.CENTER,1.,1.);
  }
  
  public boolean action(Event e, Object o){
    if(e.target == buttonOK){
      this.hide();
      Toolkit.getDefaultToolkit().sync();      
      notifyParent.postEvent(new Event(this,Event.ACTION_EVENT,null));
      //this.dispose();
      return true;
    } else if(e.target == buttonCancel){
      this.hide();
      return true;
    }
    return false;
  }  
}
