package PVS.POVTextureEditor;
// ColorMapPanel.java - a Java colors map panel
//
// Copyright (C) 1996 by Vladimir Bulatov <V.Bulatov@ic.ac.uk>.  
//             All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.

import java.awt.*;

import java.util.Vector;
import java.util.Observer;
import java.util.Observable;

import java.io.*;
import PVS.Utils.*;

class ColorMapButton extends Object
{
  int left,top, width, height,right;

  int leftborder,rightborder;

  ColorMapEntry colorMapEntry;

  Color color,bcolor,dcolor;
  int depth=2; // width of 3D shadows
  int border;   // width of left and right sides
  int selection = ColorMapEntry.NONE;

  ColorMapButton(Color color,ColorMapEntry colorMapEntry){
    this(0,0,10,10,color,colorMapEntry);
  }

  ColorMapButton(int left,int top,int width,int height,
		 Color color, // background color
		 ColorMapEntry colorMapEntry // associated color map entry
		 ){
    move(left,top,width,height);
    this.color = color;
    this.bcolor = color.brighter();
    this.dcolor = color.darker();

    this.colorMapEntry = colorMapEntry;
  }

  void move(int left, int top, int width, int height){
    this.left = left;
    this.top = top;
    this.width = width;
    this.height = height;

    right = left+width;
    border = 6;//(width-2*depth)/4; 
    leftborder = left+border;  
    rightborder = right-border;    
  }
  
  int wherePointer(int x){
    if(x < left | x > right)
      return ColorMapEntry.NONE;
    if(x < leftborder)
      return ColorMapEntry.LEFT;
    if(x < rightborder)
      return ColorMapEntry.BOTH;
    return ColorMapEntry.RIGHT;
  }

  void draw(Graphics g){
    g.setColor(color);
    g.fillRect(left+depth,top+depth,width-2*depth,height-2*depth);
    drawBorder(g,left,top,width,height,depth,bcolor,dcolor);

    switch(selection){
    case ColorMapEntry.NONE:  // draw recessed parts 
      break;
    case ColorMapEntry.LEFT:
      drawBorder(g,left+depth,top+depth,border,height-2*depth,
		 depth,dcolor,bcolor);
      break;
    case ColorMapEntry.BOTH:
      drawBorder(g,leftborder,top+depth,rightborder-leftborder,height-2*depth,
		 depth,dcolor,bcolor);
      break;
    case ColorMapEntry.RIGHT:
      drawBorder(g,rightborder,top+depth,border,height-2*depth,
		 depth,dcolor,bcolor);      
      break;
    }
  }  

  void drawBorder(Graphics g, int x, int y, int w, int h, int b, 
		  Color c1, Color c2){
    g.setColor(c1);
    for(int j=0; j < b; j++) {
      g.drawLine(x+j,y+j,x+j,y+height-j); // left
      g.drawLine(x+j,y+j,x+w-j-1,y+j); // top
    }
    g.setColor(c2);
    for(int j=0; j < b; j++) {
      g.drawLine(x+w-j-1,y+h-j-1, x+w-j-1,y+j);// right
      g.drawLine(x+w-j-1,y+h-j-1, x+j,y+h-j-1); //bottom
    }    
  }
}

class ColorMapControls extends Canvas
{  

  ColorMap colorMap = null;
  Vector buttons = null;
  int Width = -1, Height = -1;

  int cursors[] = {Frame.DEFAULT_CURSOR,Frame.W_RESIZE_CURSOR,
		   Frame.E_RESIZE_CURSOR,Frame.MOVE_CURSOR};

  ColorMapPanel colorMapPanel; // our parent

  ColorMapControls(ColorMap colorMap,ColorMapPanel panel){
    this.colorMap = colorMap;
    this.colorMapPanel = panel;
  }
  
  public Dimension preferredSize(){
    return new Dimension(256,20);
  }

  public Dimension minimumSize(){
    return preferredSize();
  }
  

  void init(ColorMap c){
    colorMap = c;
    buttons = null;
    initButtons();
    repaint();
  }

  void initButtons(){
    if(buttons == null){
      buttons = new Vector();
      for(int i=0; i< colorMap.size();i++){
	ColorMapEntry ce = colorMap.getEntry(i);      
	int start = (int)(ce.t1*Width);
	int width = (int)((ce.t2-ce.t1)*Width);
	buttons.addElement(new ColorMapButton(getBackground(),ce));
	moveButton((ColorMapButton)buttons.elementAt(i));
      }
    } else { // just move them
      for(int i=0; i< buttons.size();i++){
	moveButton((ColorMapButton)buttons.elementAt(i));
      }      
    }
    
  }

  // moves button in place
  void moveButton(ColorMapButton button){ 
    ColorMapEntry ce = button.colorMapEntry;
    int start = (int)(ce.t1*Width);
    int width = (int)((ce.t2-ce.t1)*Width);
    button.move(start,0,width,Height);
  }

  public void update(Graphics g){
    paint(g);
  }
  
  Image memImage = null;
  Graphics memGraphics = null;

  public void paint(Graphics g) {
    if(Width != size().width || Height != size().height){
      Width = size().width; Height = size().height;
      initButtons();
      memImage = this.createImage(Width,Height);      
      memGraphics = memImage.getGraphics();
    }

    memGraphics.setColor(Color.black);
    memGraphics.fillRect(0,0,Width,Height);
    
    for(int i=0; i < buttons.size(); i++){
      ((ColorMapButton)buttons.elementAt(i)).draw(memGraphics);
    }   
    g.drawImage(memImage,0,0,null);    
  }  

  int newState = ColorMapEntry.NONE;
  ColorMapButton selectedButton = null,newButton = null;

  public boolean mouseMove(Event e, int x, int y)
    {
      newState = ColorMapEntry.NONE;
      for(int i=buttons.size()-1; i >=0; i--){
	ColorMapButton b = (ColorMapButton)buttons.elementAt(i);
	int s = b.wherePointer(x);
        if(s != ColorMapEntry.NONE){
	  newState = s;
	  break;   // we point to something
	}
      }          
      ((Frame)WindowUtils.getMainWindow(this)).setCursor(cursors[newState]);
      return true;
    }

  int xDown=0,yDown=0;
  long firstDown = 0;
  ColorMapEntry colorMapEntryDown = null;
  public boolean mouseDown(Event e, int x, int y)
    {
      boolean doubleClick = false;
      if(e.when - firstDown < 500){ // double click ?
	doubleClick = true;
      }
      firstDown = e.when; // for double click check      
      
      int state = ColorMapEntry.NONE;
      ColorMapButton button = null;
      for(int i=buttons.size()-1; i >=0; i--){
	ColorMapButton b = (ColorMapButton)buttons.elementAt(i);
	int s = b.wherePointer(x);
        if(s != ColorMapEntry.NONE){
	  state = s;
	  button = b;
	  break;   // we point to something
	}
      }          

      Graphics g = getGraphics();
      if(selectedButton != null){
	selectedButton.selection = ColorMapEntry.NONE;
	selectedButton.draw(g);
      }

      if(button != null){
	// pulling button to end of the list
	//buttons.removeElement(button); 
	//buttons.addElement(button);
	button.selection = state;
	if(!doubleClick)
	  button.draw(g);	
	colorMap.setActiveEntry(button.colorMapEntry,button.selection);
	colorMapEntryDown = (ColorMapEntry)(button.colorMapEntry.clone());
      } else {
	colorMap.setActiveEntry(null,ColorMapEntry.NONE);	
      }

      selectedButton = button;
      if(doubleClick){
	if(button == null){ // create new button
	  ColorMapEntry ce = colorMap.createEntry(((double)x)/Width);
	  int position  = colorMap.addEntry(ce);
	  buttons.insertElementAt(new ColorMapButton(getBackground(),ce),position);
	  initButtons();
	  repaint();
	  colorMapPanel.colorMapCanvas.isValidImage = false;
	  colorMapPanel.colorMapCanvas.repaint();
	  selectedButton = null;
	} else {     // expand existing button
	  int i = buttons.indexOf(button);
	  if(i == 0)
	    button.colorMapEntry.t1 = 0;
	  else 
	    button.colorMapEntry.t1 = ((ColorMapButton)buttons.elementAt(i-1)).
	      colorMapEntry.t2;
	  if(i == buttons.size()-1)
	    button.colorMapEntry.t2 = 1;
	  else 
	    button.colorMapEntry.t2 = ((ColorMapButton)buttons.elementAt(i+1)).
	      colorMapEntry.t1;
	  moveButton(button);
	  button.draw(g);
	}
      firstDown = 0; // clear previous Down flag
      }
      xDown = x;
      yDown = y;
      return true;
    }

  public boolean mouseDrag(Event e, int x, int y)
    {
      if(selectedButton != null){
	colorMap.move(colorMapEntryDown,((double)x-xDown)/Width);
	initButtons();
	repaint();
	colorMapPanel.colorMapCanvas.isValidImage = false;
	colorMapPanel.colorMapCanvas.repaint();

      }
      return true;
    }

  public boolean mouseUp(Event e, int x, int y) {
    if(selectedButton != null){
      if(e.controlDown()){
	buttons.removeElement(selectedButton);
	colorMap.removeEntry(selectedButton.colorMapEntry);

	colorMapPanel.colorMapCanvas.isValidImage = false;
	colorMapPanel.colorMapCanvas.repaint();
	selectedButton = null;
	repaint();
      } else if( (e.modifiers & Event.META_MASK) != 0){ //right mouse button really
	// we will divide our button in two
	double t= (double)x/Width,t1,t2;
	t1 = selectedButton.colorMapEntry.t1;
	t2 = selectedButton.colorMapEntry.t2;
	ColorMapEntry ce = selectedButton.colorMapEntry;
	if(t > t1 + ColorMapEntry.MINIMAL_WIDTH && 
	   t < t2 - ColorMapEntry.MINIMAL_WIDTH){
	  ColorExt cright = ce.c2; // keep color to use on right end
	  ce.c2 = new ColorExt(ce.getColorExt(t));	  
	  ce.t2 = t;  // move right end 
	  //
	  // create new entry
	  //
	  ColorMapEntry cnew = colorMap.createEntry(t+ColorMapEntry.MINIMAL_WIDTH);
	  //
	  // correction of right color, which was chosen automatically
	  //
	  cnew.c2 = cright; 
	  int position  = colorMap.addEntry(cnew);
	  buttons.insertElementAt(new ColorMapButton(getBackground(),cnew),
				  position);
	  initButtons();
	  repaint();
	  colorMapPanel.colorMapCanvas.isValidImage = false;
	  colorMapPanel.colorMapCanvas.repaint();

	}
      } else {
	colorMapPanel.colorEditor.setColor(colorMap.getCurrentColor());
      }
    }
    return true;
  }
  
  public boolean mouseExit(Event e, int x, int y) {
      ((Frame)WindowUtils.getMainWindow(this)).
	setCursor(cursors[ColorMapEntry.NONE]);
      newState = ColorMapEntry.NONE;
      newButton = null;
      return true;
    } 
  
  public boolean handleEvent(Event e) {
    //System.out.println("Event:"+e.toString());
    switch(e.id){
    default:
      break;
    case Event.KEY_RELEASE:
    }
    return super.handleEvent(e);
  }
  
}

public class ColorMapPanel extends Panel implements Observer
{  

  ColorMap colorMap = null;
  ColorMapCanvas colorMapCanvas = null;
  ColorMapControls colorMapControls = null;
  GridBagLayout layout = new GridBagLayout();
  ColorEditor colorEditor;

  Button printButton = new Button("print");
  Button readButton = new Button("read");
  Button loadButton = new Button("load");
  Button saveButton = new Button("save");
  Button copyButton = new Button("copy");
  Button pasteButton = new Button("paste");
  

  ReadColorMapDialog readColorMapDialog = null;
  FileDialog file_load_dialog = null,file_save_dialog = null;
  
  ColorMapPanel(ColorMap colorMap, ColorEditor colorEditor){

    this.colorMap = colorMap;    
    this.colorEditor = colorEditor;    
    colorEditor.colorModel.addObserver(this);
    this.setLayout(layout);
    colorMapCanvas = new ColorMapCanvas(colorMap);
    colorMapControls = new ColorMapControls(colorMap,this);

    Panel pColorMapCanvas = new BorderPanel(BorderPanel.RECESSED); 
    pColorMapCanvas.setLayout(layout);
    Panel pColorMapControls = new BorderPanel(BorderPanel.RECESSED); 
    pColorMapControls.setLayout(layout);
    Panel pColorMapButtons = new BorderPanel(BorderPanel.RECESSED); 
    pColorMapButtons.setLayout(layout);

    WindowUtils.constrain(this,pColorMapCanvas,0,0,1,1,
			  GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.,0,0,0,0);
    WindowUtils.constrain(pColorMapCanvas,colorMapCanvas,0,0,1,1,
			  GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.,0,0,0,0);

    WindowUtils.constrain(this,pColorMapControls,0,1,1,1,
			  GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,0.,0,0,0,0);
    WindowUtils.constrain(pColorMapControls,colorMapControls,0,0,1,1,
			  GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,1.,0.,0,0,0,0);

    WindowUtils.constrain(this,pColorMapButtons,1,0,1,2,
			  GridBagConstraints.VERTICAL,
                          GridBagConstraints.CENTER,0.,1.,0,0,0,0);

    WindowUtils.constrain(pColorMapButtons,copyButton,0,0,1,1,
			  GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,0.2,1.);        
    WindowUtils.constrain(pColorMapButtons,pasteButton,0,1,1,1,
			  GridBagConstraints.HORIZONTAL,
                          GridBagConstraints.CENTER,0.2,1.);        
    
    WindowUtils.constrain(pColorMapButtons,printButton,0,2,1,1,
			  GridBagConstraints.NONE,
			  GridBagConstraints.CENTER,0.2,1.);        
    /*
       WindowUtils.constrain(pColorMapButtons,saveButton,0,1,1,1,
                             GridBagConstraints.NONE,
                             GridBagConstraints.CENTER,0.,1.);        
       WindowUtils.constrain(pColorMapButtons,readButton,1,0,1,1,
                             GridBagConstraints.NONE,
                             GridBagConstraints.CENTER,0.,1.);        
       WindowUtils.constrain(pColorMapButtons,loadButton,1,1,1,1,
                             GridBagConstraints.NONE, 
                             GridBagConstraints.CENTER,0.,1.);        
       */
  }    

  public void init(POVDeclare declare){
    init((POVColorMap)declare.o);
    Frame f = WindowUtils.getMainWindow(this);
    if(f != null){
      f.setTitle("ColorMap: "+declare.name);
      f.show();
    }
    f = WindowUtils.getMainWindow(colorEditor);
    if(f != null){
      f.show();
    }
  }

  public void init(ColorMap c){
    if(c != null){
      colorMap = c;
      colorMapCanvas.init(c);
      colorMapControls.init(c);	
    }
  }

  // called by ColorEditor.colorModel when color was chaged
  public void update(Observable o, Object arg){ 
    //System.out.println("ColorPanel: update");
    colorMap.setColor((ColorExt)arg);
    colorMapCanvas.isValidImage = false;
    colorMapCanvas.repaint();    
  }

  public boolean action(Event e,Object o) {
    if(e.target == printButton){
      doPrint();
      return true;
    } else if(e.target == readButton){
      doRead();
      return true;
    } else if(e.target == loadButton){
      doLoad();
      return true;
    } else if(e.target == saveButton){
      doSave(); 
      return true;
    } else if(e.target == copyButton){
      doCopy(); 
      return true;
    } else if(e.target == pasteButton){
      doPaste(); 
      return true;
    } else if(e.target == readColorMapDialog){ // user said OK in the read dialog
      doReadColorMap(); 
      return true;
    } else {
      System.out.println("action:" + e);      
    }
    return false;
  }

  void doCopy(){
    Clipboard.put(new ColorMap(this.colorMap));
  }

  void doPaste(){
    ColorMap cm = (ColorMap)Clipboard.get("ColorMap");
    if(cm != null){
      this.colorMap.assign(cm); 
      this.init(this.colorMap);
      return;
    }
    POVColorMap pcm = (POVColorMap)Clipboard.get("POVColorMap");
    if(pcm != null){
      this.colorMap.assign(pcm); 
      this.init(this.colorMap);
      return;
    }
    POVColor color = (POVColor)Clipboard.get("POVColor");
    if(color != null){
      update(null,color);
      return;
    }    
  }
  
  void doRead(){
    if(readColorMapDialog == null){
      readColorMapDialog = new 
	ReadColorMapDialog((Frame)(WindowUtils.getMainWindow(this)),
			   "ColorMap input",false,this);
      readColorMapDialog.pack();
    }
    readColorMapDialog.show(); // it doesn't blocks here :-(  
  }

  void doPrint(){
    //System.out.println(colorMap.toString());
    Frame frame = new DestroyableFrame("ColorMap output");
    TextArea text = new TextArea(colorMap.toString());
    text.setFont(new Font("Dialog",Font.PLAIN,14));
    
    frame.add("Center",text);
    frame.pack(); frame.show();
  }

  void doLoad(){
    if(file_load_dialog == null){
      file_load_dialog = new FileDialog(WindowUtils.getMainWindow(this), 
					"Load Color Map", FileDialog.LOAD);
    }
    file_load_dialog.pack();
    file_load_dialog.show();
    
    System.out.println("file: "+file_load_dialog.getFile());
    String fname = file_load_dialog.getFile();
    String fpath = file_load_dialog.getDirectory();
    
    if(fname != null && fpath != null){  // user had selected something
      File f = new File(fpath,fname);
      byte[] buffer = new byte[(int)(f.length())];
      try {
	(new FileInputStream(f)).read(buffer);
	this.init(POVParse.getColorMap(new String(buffer,0)));
      } catch(IOException ex){
	System.out.println(ex);
      }
    }
  } 

  void doReadColorMap(){
    try { 
      this.colorMap.assign(POVParse.getColorMap(readColorMapDialog.result)); 
      this.init(this.colorMap);
    } catch(IOException ex){
      System.out.println(ex);	
    }
  }

  void doSave(){
    if(file_save_dialog == null){
      file_save_dialog = new FileDialog(WindowUtils.getMainWindow(this),
					"Save Color Map", FileDialog.SAVE);
    }
    file_save_dialog.pack();
    file_save_dialog.show();
    String fname  = file_save_dialog.getFile();
    String fpath  = file_save_dialog.getDirectory();
    if(fname != null){ // user have selected a file
      try{
	new DataOutputStream(new FileOutputStream(new File(fpath,fname))).
	  writeBytes(colorMap.toString()); 
      } catch(IOException ex){
	System.out.println(ex);
      }
    }
  }
}

class ReadColorMapDialog extends Dialog{
  
  TextArea text = new TextArea();
  Button buttonOK = new Button("Read");
  Button buttonCancel = new Button("Cancel");
  String result = null;
  Component notifyParent = null;
  
  ReadColorMapDialog(Frame frame,String title,boolean modal,Component notifyParent){
    super(frame,title,modal);
    
    this.notifyParent = notifyParent;
    //text.setFont(new Font("Courier",Font.PLAIN,14));
    text.setFont(new Font("Dialog",Font.PLAIN,14));
    
    this.setLayout(new GridBagLayout());

    WindowUtils.constrain(this,text,0,0,2,1,
			  GridBagConstraints.BOTH,
			  GridBagConstraints.CENTER,1.,1.,0,0,0,0);        
    WindowUtils.constrain(this,buttonOK,0,1,1,1,
			  GridBagConstraints.NONE,
			  GridBagConstraints.CENTER,1.,0.,0,0,0,0);        
    WindowUtils.constrain(this,buttonCancel,1,1,1,1,
			  GridBagConstraints.NONE,
			  GridBagConstraints.CENTER,1.,0.,0,0,0,0);        
  }
  
  public boolean action(Event e, Object o){
    if(e.target == buttonOK){
      result = text.getText();
      notifyParent.postEvent(new Event(this,Event.ACTION_EVENT,null));
      this.hide();
      //this.dispose();
      return true;
    } else if(e.target == buttonCancel){
      this.hide();
      return true;
    }
    return false;
  }  
}
