package PVS.POVTextureEditor;
// ColorMap.java - a Java ColorMap class
//
// Copyright (C) 1996 by Vladimir Bulatov <V.Bulatov@ic.ac.uk>.  
//                  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.


import java.awt.*;
import java.awt.image.*;
import java.util.Vector;
import java.io.*;

import PVS.Utils.Fmt; // numbers formatting package

public class ColorMap extends Object {

  ColorExt dummyColor = new ColorExt();
  /**
    color used for drawing partially transparent maps
   */
  ColorExt backColor = new ColorExt(0.5,0.5,0.5,1);

  ColorMapEntry activeEntry = null;

  Vector entries = new Vector();

  public static final int  MODE_POV = 0,PVS_MODE=1;

  public int outputMode = MODE_POV;

  public ColorMap(){
    
  }

  public ColorMap(ColorMap cm){
    for(int i=0; i < cm.size(); i++){
      addEntry((ColorMapEntry)cm.getEntry(i).clone());
    }
  }    
  
  public Object clone(){
    return new ColorMap(this);
  }

  static ColorMap CreateRainbow(){
    ColorMap cm = new ColorMap();
    double c[][] = {{140/255.,6/255.,205/255.}, // violet
		 {20/255.,5/255.,205/255.},  // blue
		 {4/255.,136/255.,205/255.}, // cyan
		 {10/255.,186/255.,67/255.}, // green
		 {179/255.,186/255.,10/255.}, // yellow
		 {224/255.,129/255.,7/255.}, // orange
		 {233/255.,42/255.,18/255.}  // red    
		      };
    for(int i = 0; i < c.length; i++){
      ColorMapEntry ce = new ColorMapEntry(new ColorExt(c[i][0],c[i][1],c[i][2],0),
					   new ColorExt(c[i][0],c[i][1],c[i][2],0),
					   (double)i/c.length,
					   (i+1.0)/c.length);
      cm.addEntry(ce);
    }
    return cm;
  }


  int size(){
    return entries.size();
  }

  ColorMapEntry getEntry(int i){
    return ((ColorMapEntry)(entries.elementAt(i)));
  }

  ColorExt getColor(double t){
    for(int i=0;i<size();i++){
      ColorMapEntry cme = getEntry(i);
      if(t >= cme.t1 && t <= cme.t2)
	return cme.getColorExt(t);
    }
    return new ColorExt(0,0,0,0);
  }

  int addEntry(ColorMapEntry ce) {
    int position = -1;
    for(int i=0;i < entries.size();i++){
      ColorMapEntry c = (ColorMapEntry)entries.elementAt(i);
      if(c.t1 >= ce.t2){
	position = i;
	break;
      }
    }
    if(position < 0){
      position = entries.size();
    }
    entries.insertElementAt(ce,position);
    return position;
  }
  
  ColorMapEntry createEntry(double t){    
    if(entries.size() == 0){
      return new ColorMapEntry(new ColorExt(),new ColorExt(),0,1);
    }
    if(t <  ((ColorMapEntry)entries.firstElement()).t1){
      ColorMapEntry c = (ColorMapEntry)entries.firstElement();
      return new ColorMapEntry(new ColorExt(c.c1), new ColorExt(c.c1),0,c.t1); 
    } else if( t > ((ColorMapEntry)entries.lastElement()).t2){
      ColorMapEntry c = (ColorMapEntry)entries.lastElement();
      return new ColorMapEntry(new ColorExt(c.c2), new ColorExt(c.c2),c.t2,1);
    } 
    if(entries.size() == 1){ // case already checked
      return null;
    }

    for(int i=0;i < entries.size()-1; i++){
      ColorMapEntry c1 = (ColorMapEntry)entries.elementAt(i);
      ColorMapEntry c2 = (ColorMapEntry)entries.elementAt(i+1);
      if(t > c1.t2 && t < c2.t1){
	return new ColorMapEntry(new ColorExt(c1.c2), 
				 new ColorExt(c2.c1),c1.t2,c2.t1);      
      }      
    }  
    return null;
  }
  
  void removeEntry(ColorMapEntry ce) {
    entries.removeElement(ce);
  }

  ColorExt getCurrentColor(){
    if(activeEntry != null)
      return activeEntry.getColorExt();
    else 
      return dummyColor;
  } 
 
  void setColor(ColorExt color){
    if(activeEntry != null)
      activeEntry.setColor(color);
  }

  void setActiveEntry(ColorMapEntry entry,int what){
    activeEntry = entry;
    if(activeEntry != null)
      activeEntry.setActive(what);
  }
  
  double getPosition(){
    if(activeEntry != null){
      return activeEntry.getPosition();
    }
    return 0.0;
  }
  
  public String toString(){
    ByteArrayOutputStream out = new ByteArrayOutputStream();
    PrintStream prn = new PrintStream(out); // will print to byte array
    if(outputMode == MODE_POV){
      prn.println("color_map {");
      if(entries.size() == 0){
	System.err.println("Error: empty ColorMap");
      }
      for(int i=0;i < entries.size(); i++){
	ColorMapEntry c = (ColorMapEntry)entries.elementAt(i);
	prn.print("  [" + Fmt.fmt(c.t1,7,5) + 
		  ", "  + Fmt.fmt(c.t2,7,5) + " color rgbf");
	prn.println("< " + Fmt.fmt(c.c1.r,7,5) +
		    ", " + Fmt.fmt(c.c1.g,7,5)+
		    ", " + Fmt.fmt(c.c1.b,7,5) +
		    ", " + Fmt.fmt(c.c1.f,7,5)+">");
	prn.println("                    color rgbf" +
		    "< " + Fmt.fmt(c.c2.r,7,5) + 
		    ", " + Fmt.fmt(c.c2.g,7,5)+
		    ", " + Fmt.fmt(c.c2.b,7,5)+
		    ", " + Fmt.fmt(c.c2.f,7,5) + ">]");
      }  
      prn.println("}");
    }
    prn.flush();
    return out.toString();
  }
  
  
  void move(ColorMapEntry entry, double dx){
    if(activeEntry != null){
      double left = 0, right = 1;
      int i = entries.indexOf(activeEntry);
      if(i>0){ // not first element
	left =  ((ColorMapEntry)(entries.elementAt(i-1))).t2;
      } 
      if(i < entries.size()-1){// not last element
	right =  ((ColorMapEntry)(entries.elementAt(i+1))).t1;      
      }
      switch(activeEntry.active){
      default:
	break;
      case ColorMapEntry.BOTH:
	{
	  double t1 = entry.t1 + dx, t2 = entry.t2 + dx;
	  if(t1 < left){
	    t1 = left; t2 = left + entry.t2 - entry.t1;
	  } else if(t2 > right){
	    t2 = right;
	    t1 = right - (entry.t2 - entry.t1);
	  }
	  activeEntry.t1 = t1;
	  activeEntry.t2 = t2;
	}
	break;

      case ColorMapEntry.LEFT:
	{
	  double t1 = entry.t1 + dx;
	  if(t1 < left){
	    t1 = left; 
	  } else if(t1 > entry.t2 - ColorMapEntry.MINIMAL_WIDTH){
	    t1 = entry.t2 - ColorMapEntry.MINIMAL_WIDTH;
	  }
	  activeEntry.t1 = t1;
	}
	break;

      case ColorMapEntry.RIGHT:
	{
	  double t2 = entry.t2 + dx;
	  if(t2 > right){
	    t2 = right; 
	  } else if(t2 < entry.t1 + ColorMapEntry.MINIMAL_WIDTH){
	    t2 = entry.t1 + ColorMapEntry.MINIMAL_WIDTH;
	  }
	  activeEntry.t2 = t2;
	}
	break;
      }      
    }
  }
 

  Image createStrip(int width){
    int[] Pixels = null;
    return createStrip(width,Pixels);
  }

  Image createStrip(int width, int[] Pixels){
    if(Pixels == null) 
      Pixels = new int[3*width];
    for(int i=0;i<3*width;i++)
      Pixels[i] = 0xff000000;  // to make Pixels not transparent
    for(int i=0; i < size(); i++){
      ColorMapEntry ce = getEntry(i);
      int start = (int)Math.floor(ce.t1*width);
      int end = (int)Math.ceil(ce.t2*width);
      start = Math.min(Math.max(0,start),width);
      end = Math.min(Math.max(0,end),width);

      for(int j = start; j < end; j++){
	Pixels[j+width] = Pixels[j] = ce.getPixel((double)j/width);
	Pixels[j+2*width] = (ce.getFullColor((double)j/width,
					     ColorExt.black)). getRGB();	
      }
    }   
    if(ImageTGA.doGamma)
      ImageTGA.GammaImage(Pixels);
    ImageProducer ip = new MemoryImageSource( width,3, Pixels, 0, width );
    return Toolkit.getDefaultToolkit().createImage(ip);    
  }  

  void assign(ColorMap cmap){
    this.entries = cmap.entries;
    this.activeEntry = null;
  }

}

