package PVS.POVTextureEditor;
// ColorEditor.java - a Java colors editor
//
// Copyright (C) 1996 by Vladimir Bulatov <V.Bulatov@ic.ac.uk>.  
//        All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.


import java.awt.*;
import java.awt.image.MemoryImageSource;
import java.util.Observable;

import PVS.Utils.*;


class ColorPanel extends Canvas {

  ColorModel colorModel = null;

  public ColorPanel(ColorModel colorModel) {
    this.colorModel = colorModel;
  }

  public Dimension preferredSize(){
    return new Dimension(80,60);
  }

  public Dimension minimumSize(){
    return preferredSize();
  }

  public void update(Graphics g){
    paint(g);
  }

  public void paint(Graphics g) {
    int[] bytes = {colorModel.color.getRGB(),colorModel.color.getRGB(),
		   colorModel.color.getFilteredColor(ColorExt.black).getRGB()};
    if(ImageTGA.doGamma)
      ImageTGA.GammaImage(bytes);
    g.drawImage(this.createImage(new MemoryImageSource(1,3,bytes,0,1)),
		0,0,size().width,size().height,null);
  }  
  
}

class ColorArea extends Canvas {

  ColorModel colorModel = null;
  ColorEditor parent;
  public ColorArea(ColorModel colorModel,ColorEditor parent) {
    this.colorModel = colorModel;
    this.parent = parent;
    makeImage();    
  }

  public Dimension preferredSize(){
    return new Dimension(256,256);
  }

  public Dimension minimumSize(){
    return new Dimension(128,128);
  }

  /*
  int Width=-1, Height=-1;
  Image memImage = null;
  Graphics memGraphics = null;
  void checkImage(){
    if(size().width != Width || size().height != Height ){
      Width = size().width; Height = size().height;     
      memImage = this.createImage(Width,Height);
      memGraphics = memImage.getGraphics();
    }    
  }
*/

  Image memImage = null;
  Image[] memImages = null;

  //
  // work around some bug in Motif, 
  // which result in crushes of Netscape in Linux on stretched images  
  boolean workAroundNetscapeCrush = true;

  void makeImage(){
    
    int length = 256 >> colorModel.grid;
    
    int[] bytes = new int[length*length];
    if(workAroundNetscapeCrush){
      memImages = new Image[length]; 
    }
    int grid = colorModel.grid;
    for (int i = 0; i < length; i++){
      int offset = i*length;
      switch(colorModel.model){
      case ColorModel.R:
	int red = colorModel.color.getRed();
	for (int j = 0; j < length; j++)
	  bytes[j+offset] = ColorExt.getRGB(red,(i<<grid),(j<<grid));
	break;
      case ColorModel.G: 
	int green = colorModel.color.getGreen();
	for (int j = 0; j < length; j++)
	  bytes[j+offset] = ColorExt.getRGB((i<<grid),green,(j<<grid));
	break;
      case ColorModel.B:
	int blue = colorModel.color.getBlue();
	for (int j = 0; j < length; j++)
	  bytes[j+offset] = ColorExt.getRGB((i<<grid),(j<<grid),blue);
	break;
      case ColorModel.H:
	double h = colorModel.hsl[0];
	for (int j = 0; j < length; j++)
	  bytes[j+offset] = ColorExt.HSBtoRGB(h, (i<<grid)/255.0, (j<<grid)/255.0);
	break;
      case ColorModel.S:
	double s = colorModel.hsl[1];
	for (int j = 0; j < length; j++)
	  bytes[j+offset] = ColorExt.HSBtoRGB((i<<grid)/255.0, s, (j<<grid)/255.0);
	break;
      case ColorModel.L:
	double l = colorModel.hsl[2];      
	for (int j = 0; j < length; j++)
	  bytes[j+offset] = ColorExt.HSBtoRGB((i<<grid)/255.0, (j<<grid)/255.0,l);
	break;
      }
      if(workAroundNetscapeCrush){    
	if(ImageTGA.doGamma)
	  ImageTGA.GammaImage(bytes,offset,length);
	memImages[i] = this.createImage(new MemoryImageSource(length,1,bytes,offset,length));
      }
    }
    if(!workAroundNetscapeCrush){
      if(ImageTGA.doGamma)
	ImageTGA.GammaImage(bytes);
      memImage = this.createImage(new MemoryImageSource(length,length,bytes,0,length));
    }
  }
  
  public void redraw(){
    makeImage(); repaint();
  }

  public void update(Graphics g){
    paint(g);
  }

  public void paint(Graphics g) 
    {
      if(workAroundNetscapeCrush){
	for(int i=0;i < memImages.length;i++){
	  int y0 = (i*size().height)/memImages.length,
	  y1 = ((i+1)*size().height)/memImages.length;
	  if(y1-y0 > 0)
	    g.drawImage(memImages[i],0, y0, size().width, y1-y0, null);
	}
      } else {      
	g.drawImage(memImage,0,0,size().width,size().height,null);
      }
      
      int xcol=0,ycol=0;
      switch(colorModel.model){
      default:
      case ColorModel.R:
	ycol = colorModel.color.getGreen();
	xcol = colorModel.color.getBlue();
	break;
      case ColorModel.G:
	ycol = colorModel.color.getRed();
	xcol = colorModel.color.getBlue();
	break;
      case ColorModel.B:
	ycol = colorModel.color.getRed();
	xcol = colorModel.color.getGreen();
	break;
      case ColorModel.H:
	ycol = (int)(colorModel.hsl[1]*255);
	xcol = (int)(colorModel.hsl[2]*255);
	break;
      case ColorModel.S:
	ycol = (int)(colorModel.hsl[0]*255);
	xcol = (int)(colorModel.hsl[2]*255);
	break;
      case ColorModel.L:
	ycol = (int)(colorModel.hsl[0]*255);
	xcol = (int)(colorModel.hsl[1]*255);
	break;
      }

      xcol = (int)((xcol/255.0)*size().width);
      ycol = (int)((ycol/255.0)*size().height);


      //Draw color location
      g.setColor(Color.white);
      g.drawOval(xcol-2,ycol-2,5,5);
      
      g.setColor(Color.black);
      g.drawOval(xcol-3,ycol-3,7,7);
            
    }

  void makeColor(int x,int y){
    if(y < 0) y = 0;
    if(y > size().height-1) y =  size().height-1;
    if(x < 0) x = 0;
    if(x > size().width-1) x =  size().width-1;
    double xx,yy;
    xx = ((double)x)/size().width;
    yy = ((double)y)/size().height;
    ColorExt old = colorModel.color;
    switch(colorModel.model){
    case colorModel.R:
      colorModel.setColor(new ColorExt(old.r, yy, xx,old.f));break;
    case colorModel.G:
      colorModel.setColor(new ColorExt(yy, old.g, xx,old.f));break;
    case colorModel.B:
      colorModel.setColor(new ColorExt(yy,xx,old.b));break;
    case colorModel.H:
      colorModel.setColor(ColorExt.getHSBColor(colorModel.hsl[0], yy, xx,old.f));break;
    case colorModel.S:
      colorModel.setColor(ColorExt.getHSBColor( yy,colorModel.hsl[1], xx,old.f));break;
    case colorModel.L:
      colorModel.setColor(ColorExt.getHSBColor( yy, xx,colorModel.hsl[2],old.f));break;
    }
  }  

  public boolean mouseDrag(Event e, int x, int y){
    makeColor(x,y);
    parent.colorPanel.repaint();

    parent.colorSlider.makeImage();
    parent.colorSlider.repaint();

    return false;
  }

  public boolean mouseUp(Event e, int x, int y){
    makeColor(x,y);
    parent.colorPanel.repaint();
    parent.colorSlider.makeImage();
    parent.colorSlider.repaint();
    parent.initValues();
    repaint();
    return false;
  }

  public boolean mouseExit(Event e, int x, int y) {
    ((Frame)WindowUtils.getMainWindow(this)).setCursor(Frame.DEFAULT_CURSOR);
    return true;
  }    
  
  public boolean mouseEnter(Event e, int x, int y) {
    ((Frame)WindowUtils.getMainWindow(this)).setCursor(Frame.HAND_CURSOR);
    return true;
  }  
}

class ColorSlider extends Canvas {

  ColorModel colorModel;
  ColorEditor parent;  

  public ColorSlider(ColorModel colorModel, ColorEditor parent) {
    this.colorModel = colorModel;
    makeImage();    
    this.parent = parent;
  }

  public Dimension preferredSize(){
    return new Dimension(40,256);
  }

  public Dimension minimumSize(){
    return preferredSize();
  }

  Image memImage = null;
  void makeImage(){
    int length = (256 >> colorModel.grid);

    int bytes[] = new int[length];
    int c = 0, shift = 0;
    switch(colorModel.model){
    case ColorModel.R:
      c = ColorExt.getRGB(0,colorModel.color.getGreen(),colorModel.color.getBlue());
      shift = colorModel.grid + 16;
      for (int i = 0; i < length; i++)
	bytes[i] = (i<<shift) | c;
      break;
    case ColorModel.G:
      c = ColorExt.getRGB(colorModel.color.getRed(),0,colorModel.color.getBlue());
      shift = colorModel.grid + 8;
      for (int i = 0; i < length; i++)
	bytes[i] = (i<<shift) | c;
      break;
    case ColorModel.B:
      c = ColorExt.getRGB(colorModel.color.getRed(),colorModel.color.getGreen(),0);;
      shift = colorModel.grid;
      for (int i = 0; i < length; i++)
	bytes[i] = (i<<shift) |c;
      break;
    case ColorModel.H:
      for (int i = 0; i < length; i++)
	bytes[i] = ColorExt.HSBtoRGB((i<<colorModel.grid)/255.0,colorModel.hsl[1],colorModel.hsl[2]);
      break;
    case ColorModel.S:
      for (int i = 0; i < length; i++)
	bytes[i] = ColorExt.HSBtoRGB(colorModel.hsl[0],(i<<colorModel.grid)/255.0,colorModel.hsl[2]);
      break;
    case ColorModel.L:
      for (int i = 0; i < length; i++)
	bytes[i] = ColorExt.HSBtoRGB(colorModel.hsl[0],colorModel.hsl[1],(i << colorModel.grid)/255.0);
      break;
    }
    if(ImageTGA.doGamma)
      ImageTGA.GammaImage(bytes);
    memImage = this.createImage(new MemoryImageSource(1,length,bytes,0,1));
  }

  public void redraw(){
    makeImage(); repaint();
  }

  public void update(Graphics g){
    paint(g);
  }

  void drawposition(Graphics g){
    int shift = size().width/4;
    g.setColor(getBackground());
    g.fillRect(0,0,shift,size().height);

    double level = 0;
    switch(colorModel.model){
    case colorModel.R:
      level = colorModel.color.r;break;
    case colorModel.G:
      level = colorModel.color.g;break;
    case colorModel.B:
      level = colorModel.color.b;break;
    case colorModel.H:
      level = colorModel.hsl[0];break;
    case colorModel.S:
      level = colorModel.hsl[1];break;
    case colorModel.L:
      level = colorModel.hsl[2];break;
    }
    
    int ilevel = (int)((level)*size().height) - 1;      
    /*
       g.setColor(Color.white);
       g.fillRect(0,level-1,shift,3);
       g.setColor(Color.black);
       g.drawRect(0,level-2,shift-1,5);
       */
    int border = 2;
    int w = shift-1, h = 6;
    Color col1 = getBackground().darker();
    Color col2 = getBackground().brighter();
    g.setColor(col1);
    for(int i=0; i< border; i++) {
      g.drawLine(i,i+ilevel,w-i,i+ilevel);
      g.drawLine(i,i+ilevel,i,h-i+ilevel);
    }
    g.setColor(col2);
    for(int i=0; i<border; i++) {
      g.drawLine(w-i,h-i+ilevel, w-i,i+ilevel);
      g.drawLine(w-i,h-i+ilevel, i,h-i+ilevel);
    }
    
  }
  public void paint(Graphics g) 
    {
      int shift = size().width/4;

      //checkImage();      
      int border = 2;

      g.drawImage(memImage,shift+border, border, 
		  size().width-shift-2*border, size().height-2*border,null);
      drawposition(g);


      int w = size().width-1-shift, h = size().height-1;
      Color col1 = getBackground().darker();
      Color col2 = getBackground().brighter();
      g.setColor(col1);
      for(int i=0; i<border; i++) {
	g.drawLine(shift+i,i,shift+w-i,i);
	g.drawLine(shift+i,i,shift+i,h-i);
      }
      g.setColor(col2);
      for(int i=0; i<border; i++) {
	g.drawLine(shift+w-i,h-i, shift+w-i,i);
	g.drawLine(shift+w-i,h-i, shift+i,h-i);
      }

    }  

  void makeColor(int y){
    if(y < 0) y = 0;
    if(y > size().height-1) y =  size().height-1;
    double level = ((double)y)/size().height;
    ColorExt old = colorModel.color;
    switch(colorModel.model){
    case colorModel.R:
      colorModel.setColor(new ColorExt(level, old.g, old.b,old.f));
      break;
    case colorModel.G:
      colorModel.setColor( new ColorExt(old.r, level, old.b,old.f));
      break;
    case colorModel.B:
      colorModel.setColor( new ColorExt(old.r, old.g,level,old.f));
      break;
    case colorModel.H:
      colorModel.setColor( ColorExt.getHSBColor(level,colorModel.hsl[1],colorModel.hsl[2],old.f));
      break;
    case colorModel.S:
      colorModel.setColor( ColorExt.getHSBColor(colorModel.hsl[0],level,colorModel.hsl[2],old.f));
      break;
    case colorModel.L:
      colorModel.setColor( ColorExt.getHSBColor(colorModel.hsl[0],colorModel.hsl[1],level,old.f));
      break;
    }
  }  

  public boolean mouseUp(Event e, int x, int y){
    makeColor(y);
    parent.colorArea.redraw();
    parent.colorPanel.repaint();
    parent.initValues();
    repaint();
    return false;
  }

  public boolean mouseDrag(Event e, int x, int y){
    makeColor(y);
    parent.colorPanel.repaint();
    drawposition(this.getGraphics());
    if(parent.alwaysRepaint){
      parent.colorArea.redraw();
    }
    //repaint();
    return false;
  }

  public boolean mouseExit(Event e, int x, int y) {
    ((Frame)WindowUtils.getMainWindow(this)).setCursor(Frame.DEFAULT_CURSOR);
    return true;
  }    

  public boolean mouseEnter(Event e, int x, int y) {
    ((Frame)WindowUtils.getMainWindow(this)).setCursor(Frame.HAND_CURSOR);
    return true;
  }    

}

public class ColorEditor extends Panel {
  
  GridBagLayout layout = new GridBagLayout();

  //Choice formatChoice = new Choice();
  public static final int INT = 0, FLOAT = 1, HEX = 2;
  int formatValue = FLOAT; // format of color representation

  DoubleR[] rgb = new DoubleR[3];
  DoubleR[] hsl = new DoubleR[3];
  DoubleR  filter = new DoubleR(0);

  DoubleField [] colorValue = new DoubleField[6];
  DoubleField filterField = new DoubleField("filter:",filter,layout,4);    
  FilterCanvas filterCanvas = new FilterCanvas(filterField);
  
  Checkbox[] cbColorModel = new Checkbox[6];
  String [] cbColorModelName = {"R:","G:","B:","H:","S:","L:"};
  CheckboxGroup cbgRGBColorModel = new CheckboxGroup();
  CheckboxGroup cbgHSLColorModel = new CheckboxGroup();

  ColorModel colorModel = new ColorModel(new ColorExt(),ColorModel.H,4);

  Checkbox[] cbGrid = new Checkbox[4];
  String [] cbGridName = {"x2","x4","x8","x16"};
  CheckboxGroup cbgGrid = new CheckboxGroup();

  ColorPanel colorPanel = null;
  ColorArea colorArea = null;
  ColorSlider colorSlider = null;

  TabbedPanel pNumbersPanel = null;

  boolean alwaysRepaint = false;  // do full repaint at each movie of mouse

  public ColorEditor(ColorExt color) {

    colorModel.setColor(color);

    colorPanel = new ColorPanel(colorModel);
    colorArea = new ColorArea(colorModel,this);
    colorSlider = new ColorSlider(colorModel,this);

    this.setLayout(layout);

    Panel pColorArea1 = new BorderPanel(BorderPanel.RAISED); 
    pColorArea1.setLayout(layout);
    Panel pColorArea = new BorderPanel(BorderPanel.RECESSED); 
    pColorArea.setLayout(layout);
    WindowUtils.constrain(pColorArea1,pColorArea,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.,0,0,0,0);
    WindowUtils.constrain(this,pColorArea1,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.,0,0,0,0);
    WindowUtils.constrain(pColorArea,colorArea,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.,0,0,0,0);

    Panel pColorSlider = new BorderPanel(BorderPanel.RAISED); 
    pColorSlider.setLayout(layout);
    WindowUtils.constrain(this,pColorSlider,1,0,1,1,GridBagConstraints.VERTICAL,
                          GridBagConstraints.CENTER,0.0,0.0,0,0,0,0);
    WindowUtils.constrain(pColorSlider,colorSlider,0,0,1,1,GridBagConstraints.VERTICAL,
                          GridBagConstraints.CENTER,1.,1.,0,0,0,0);

    Panel pControls = new Panel(); pControls.setLayout(layout);
    WindowUtils.constrain(this,pControls,2,0,1,1,GridBagConstraints.VERTICAL,
                          GridBagConstraints.CENTER,0.,0.,0,0,0,0);

    Panel pColor = new BorderPanel(BorderPanel.RAISED); pColor.setLayout(layout);
    Panel pColor1 = new BorderPanel(BorderPanel.RECESSED); pColor1.setLayout(layout);
    WindowUtils.constrain(pControls,pColor,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,0.5,0.5,0,0,0,0);
    WindowUtils.constrain(pColor,pColor1,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,0.5,0.5,0,0,0,0);

    WindowUtils.constrain(pColor1,colorPanel,0,0,1,1,GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.,0,0,0,0);
    

    /*       
       formatChoice.addItem("dec");formatChoice.addItem("hex");formatChoice.
       addItem("flt");
       formatChoice.select(formatValue);
       WindowUtils.constrain(pNumbers,new Label("Format:"),0,0,2,1,
       GridBagConstraints.NONE, GridBagConstraints.EAST,0.,0.,0,0,0,0);      
       WindowUtils.constrain(pNumbers,formatChoice,2,0,1,1,GridBagConstraints.NONE,
       GridBagConstraints.EAST,0.,0.,0,0,0,0);      
       */

    
    WindowUtils.constrain(pControls,createNumbersPanel(),0,2,1,1,
			  GridBagConstraints.BOTH, GridBagConstraints.WEST,1.,1.);

    Panel pFilter = new BorderPanel(BorderPanel.RAISED); pFilter.setLayout(layout);
    WindowUtils.constrain(pFilter,filterField,0,0,1,1,
			  GridBagConstraints.NONE, GridBagConstraints.CENTER,1.,1.);

    Panel pFilter1 = new BorderPanel(); pFilter1.setLayout(layout);
    WindowUtils.constrain(pFilter1,filterCanvas,0,0,1,1,
			  GridBagConstraints.BOTH, GridBagConstraints.CENTER,1.,1.);

    WindowUtils.constrain(pFilter,pFilter1,0,1,1,1,
			  GridBagConstraints.BOTH, GridBagConstraints.CENTER,1.,1.);

    WindowUtils.constrain(pControls,pFilter,0,3,1,1,
			  GridBagConstraints.BOTH, GridBagConstraints.CENTER,1.,1.);

    initValues();
  }

  Panel createGridPanel(){

    Panel pGrid = new Panel(); pGrid.setLayout(layout);

    for(int i=0;i < cbGrid.length; i++){
      cbGrid[i] = new Checkbox(null,cbgGrid,false);
      WindowUtils.constrain(pGrid,cbGrid[i],i,0,1,1,GridBagConstraints.NONE,
			    GridBagConstraints.WEST,0.,0.);      
      WindowUtils.constrain(pGrid,new Label(cbGridName[i]),i,1,1,1,
			    GridBagConstraints.NONE,
			    GridBagConstraints.WEST,0.,0.);      
    }
    cbgGrid.setCurrent(cbGrid[colorModel.grid-1]);
    return pGrid;
  }

  Panel createNumbersPanel(){

   Panel pRGB = new Panel(); pRGB.setLayout(layout);
   Panel pHSL = new Panel(); pHSL.setLayout(layout);

   for(int i=0;i < 3;i++){
     rgb[i] = new DoubleR(1);
     cbColorModel[i] = new Checkbox(null,cbgRGBColorModel,false);
     WindowUtils.constrain(pRGB,cbColorModel[i],0,i,1,1,GridBagConstraints.NONE,
			   GridBagConstraints.WEST,0.,0.);      
     colorValue[i] = new DoubleField(cbColorModelName[i],rgb[i],layout,4);
     WindowUtils.constrain(pRGB,colorValue[i],1,i,1,1,GridBagConstraints.NONE,
			   GridBagConstraints.EAST,0.,0.);           
   }
   for(int i=3;i < 6;i++){
     hsl[i-3] = new DoubleR(1);
     cbColorModel[i] = new Checkbox(null,cbgHSLColorModel,false);
     WindowUtils.constrain(pHSL,cbColorModel[i],0,i,1,1,GridBagConstraints.NONE,
			   GridBagConstraints.WEST,0.,0.);      
     colorValue[i] = new DoubleField(cbColorModelName[i],hsl[i-3],layout,4);
     WindowUtils.constrain(pHSL,colorValue[i],1,i,1,1,GridBagConstraints.NONE,
			   GridBagConstraints.EAST,0.,0.);           
   }

   cbgRGBColorModel.setCurrent(cbColorModel[0]);
   cbgHSLColorModel.setCurrent(cbColorModel[3]);
    
   pNumbersPanel = new TabbedPanel(this);
   pNumbersPanel.addItem("HSL",pHSL);pNumbersPanel.addItem("RGB",pRGB);
   return pNumbersPanel;

  }

  void redraw(){
    colorArea.redraw();
    colorSlider.redraw();
    colorPanel.repaint();
    filterCanvas.repaint();
  }

  void setColor(ColorExt color){
    colorModel.setEditingColor(color);
    initValues();
    redraw();
  }

  void init(ColorExt color){
    setColor(color);
  }

  public void init(POVDeclare declare){
    init(((POVColor)declare.o));
    Frame f = WindowUtils.getMainWindow(this);
    if(f != null){
      f.setTitle("Color: "+declare.name);
      f.show();
    }
  }

  double round (double f){
    return ((int)(f*1000+0.5))/1000.0f;
  }

  void initValues(){
    switch(formatValue){
    case FLOAT:
      colorValue[0].setValue(round(colorModel.color.r));
      colorValue[1].setValue(round(colorModel.color.g));
      colorValue[2].setValue(round(colorModel.color.b));
      colorValue[3].setValue(round(colorModel.hsl[0]));
      colorValue[4].setValue(round(colorModel.hsl[1]));
      colorValue[5].setValue(round(colorModel.hsl[2]));
      filterField.setValue(round(colorModel.color.f));
      break;
    }
  }

  public boolean action(Event e, Object what){
    if(e.target == pNumbersPanel){
      if(((String)what).equals("RGB")){
	for(int i = 0; i<3;i++){
	  if(cbgRGBColorModel.getCurrent() == cbColorModel[i]){
	    colorModel.model = i;
	  }	  
	}	
      } else {
	for(int i = 3; i < 6; i++){
	  if(cbgHSLColorModel.getCurrent() == cbColorModel[i]){
	    colorModel.model = i;
	  }	  
	}	
      }
      redraw();
      return true;
    }

    for(int i=0; i < cbColorModel.length;i++){
      if(e.target == cbColorModel[i]){
	colorModel.model = i;
	redraw();
	return true;
      }	
    }   

    for(int i=0; i < cbGrid.length;i++){
      if(e.target == cbGrid[i]){
	colorModel.grid = i+1;
	colorArea.redraw();
	return true;
      }	      
    }
    // may be somebody from text fields
    for(int i=0;i<6;i++){
      if(e.target == colorValue[i].textField){
	switch(colorModel.model){
	case ColorModel.R:
	case ColorModel.G:
	case ColorModel.B:
	  colorModel.setColor( new ColorExt(rgb[0].value, rgb[1].value,
					    rgb[2].value,filter.value));
	  break;
	case ColorModel.H:
	case ColorModel.S:
	case ColorModel.L:
	  colorModel.setColor(ColorExt.getHSBColor(hsl[0].value,hsl[1].value,
						   hsl[2].value,filter.value));
	  break;
	}
	initValues();
	redraw();
	return true;
      }
    }
    // filter canvas or filter text field 
    if(e.target == filterField.textField || e.target == filterField){
      colorModel.setColor( new ColorExt(rgb[0].value, rgb[1].value,
					rgb[2].value,filter.value));
      colorPanel.repaint(); 
      filterCanvas.repaint();
      return true;
    }
    return false;
  }
  
  public String getAppletInfo() {
    return getClass().getName() + 
      " - Copyright (C) 1996 by Vladimir Bulatov <V.Bulatov.ic.ac.uk>.\n"+
	"  All rights reserved.";
  }

  public static void main(String[] args) {
    ColorEditor applet = new ColorEditor(new ColorExt());
    Frame frame = new AppletFrame("ColorEditor Test");
    frame.add("Center", applet);
    frame.pack();
    frame.show();
  }
}
